#!/bin/sh
# geninitrd mod: cryptsetup luks
USE_LUKS=${USE_LUKS:-yes}

# true if root device is crypted with cryptsetup luks
# and we should init cryptsetup luks at boot
have_luks=no

# device to use for name for cryptsetup luks
LUKSNAME=""

# setup geninitrd module
# @access	public
setup_mod_luks() {
	cryptsetup=$(find_tool $initrd_dir/cryptsetup /sbin/cryptsetup-initrd)

	if [ ! -x /sbin/cryptsetup ] || [ ! -x "$cryptsetup" ]; then
		USE_LUKS=no
	fi
}

# return true if node is cryptsetup luks encrypted
# @param	string $node device node to be examined
# @access	public
is_luks() {
	local node="$1"

	# luks not wanted
	if is_no "$USE_LUKS"; then
		return 1
	fi

	local dev dm_name=${node#/dev/mapper/}
	if [ "$node" = "$dm_name" ]; then
		verbose "is_luks: $node is not device mapper name"
		return 1
	fi

	dev=$(awk -vdm_name="$dm_name" '$1 == dm_name { print $2 }' /etc/crypttab)
	if [ "$dev" ]; then
		dev=$(find_devname "$dev")
		/sbin/cryptsetup isLuks $dev
		rc=$?
	else
		rc=1
	fi

	if [ $rc = 0 ]; then
		verbose "is_luks: $node is cryptsetup luks"
	else
		verbose "is_luks: $node is not cryptsetup luks"
	fi
	return $rc
}

# find modules for $devpath
# @param	$devpath	device to be examined
# @access	public
find_modules_luks() {
	local devpath="$1"
	local dev=""

	LUKSNAME=${devpath#/dev/mapper/}

	find_module "dm-crypt"

	# TODO: autodetect
	find_module "-aes"
	find_module "-cbc"
	find_module "-cbc(aes)"
	find_module "-xts(aes)"
	find_module "-af-alg"
	find_module "-algif_hash"
	find_module "-algif_skcipher"
	find_module "-loop"

	# recurse
	dev=$(awk -vLUKSNAME="$LUKSNAME" '$1 == LUKSNAME { print $2 }' /etc/crypttab)
	if [ -n "$dev" ]; then
		dev=$(find_devname "$dev")
		find_modules_for_devpath $dev
		have_luks=yes
	else
		die "Cannot find '$LUKSNAME' in /etc/crypttab"
	fi
}


# generate initrd fragment for cryptsetup luks init
# @access	public
initrd_gen_luks() {
	if ! is_yes "$have_luks"; then
		return
	fi

	inst_d /bin
	inst_exec $cryptsetup /bin/cryptsetup
	inst_d /var/run/cryptsetup

	mount_dev
	mount_sys
	initrd_gen_devices
	# TODO: 'udevadm settle' is called by lukssetup, is udev optional?

	verbose "luks: process /etc/crypttab $LUKSNAME"
	luks_crypttab $LUKSNAME
}


# PRIVATE METHODS
key_is_random() {
	[ "$1" = "/dev/urandom" -o "$1" = "/dev/hw_random" -o "$1" = "/dev/random" ]
}

# produce cryptsetup from $name from /etc/crypttab
luks_crypttab() {
	local LUKSNAME="$1"

	# copy from /etc/rc.d/init.d/cryptsetup
	local dst src key opt mode owner failsafe token libdir cryptdir fido2_token_found fido2_device fido2_token_timeout fido2_token_check

	while read dst src key opt; do
		[ "$dst" != "$LUKSNAME" ] && continue

		failsafe=""

		if [ -z "$key" ] || [ "x$key" = "xnone" ] || [ "x$key" = "x-" ]; then
			failsafe=1
			key="/etc/cryptsetup-keys.d/$LUKSNAME.key"
		fi

		if test -e "$key" ; then
			mode=$(LC_ALL=C ls -l "$key" | cut -c 5-10)
			owner=$(LC_ALL=C ls -l $key | awk '{ print $3 }')
			if [ "$mode" != "------" ] && ! key_is_random "$key"; then
				die "INSECURE MODE FOR $key"
			fi
			if [ "$owner" != root ]; then
				die "INSECURE OWNER FOR $key"
			fi
		elif [ -n "$failsafe" ]; then
			key=""
		else
			die "Key file for $dst not found"
		fi

		src=$(find_devname "$src")
		if /sbin/cryptsetup isLuks "$src" 2>/dev/null; then
			if key_is_random "$key"; then
				die "$dst: LUKS requires non-random key, skipping"
			fi
			if [ "$key" ]; then
				keyfile=/etc/.$dst.key
				inst $key $keyfile
			fi
			for libdir in $(get_libdir /usr/LIBDIR); do
				if [ -d $libdir/cryptsetup ]; then
					cryptdir=$libdir/cryptsetup
					break
				fi
			done
			if ! is_no "$USE_LUKS_TOKEN" && [ -n "$cryptdir" ]; then
				for token in $(/sbin/cryptsetup luksDump $src | sed -n -e '/^Tokens:/,/^[^[:space:]]/ { /^[[:space:]]*[[:digit:]]*:/ { s/^[[:space:]]*[[:digit:]]*:[[:space:]]*// p } }'); do
					verbose "Found cryptsetup token: $token"
					case "$token" in
						systemd-fido2)
							inst_d $cryptdir
							inst_exec $cryptdir/libcryptsetup-token-$token.so $cryptdir
							inst_exec $libdir/libfido2.so.1 $libdir
							if [ -e $libdir/libpcsclite_real.so.1 ]; then
								inst_exec $libdir/libpcsclite_real.so.1 $libdir
							fi
							fido2_token_found=1
							;;
						systemd-tpm2)
							inst_d $cryptdir
							inst_exec $cryptdir/libcryptsetup-token-$token.so $cryptdir
							inst_exec $libdir/libtss2-esys.so.0 $libdir/libtss2-rc.so.0 $libdir/libtss2-mu.so.0 $libdir
							;;
						*)
							inst_d $cryptdir
							inst_exec $cryptdir/libcryptsetup-token-$token.so $cryptdir
							;;
					esac
				done
			fi

			crypttab_opt=""
			old_IFS="$IFS"
			IFS=","
			for option in $opt; do
				case "$option" in
					discard|allow-discards)
						crypttab_opt="$crypttab_opt --allow-discards"
						;;
					fido2-device=*)
						fido2_device=${option#*=}
						;;
					*)
						warn "$dst: option \'$option\' is unsupported for LUKS partitions, ignored"
						;;
				esac
			done
			IFS="$old_IFS"

			fido2_token_timeout=${FIDO2_TOKEN_TIMEOUT:-30}

			if [ -n "$fido2_token_found" ] && [ $fido2_token_timeout -gt 0 ]; then
				if [ ${fido2_device:-auto} = "auto" ]; then
					inst_exec /usr/bin/fido2-token /bin
					fido2_token_check='[ -z "$(/bin/fido2-token -L 2>/dev/null)" ]'
				else
					fido2_token_check="[ ! -e \"$fido2_device\" ]"
				fi
				add_linuxrc <<-EOF
				if $fido2_token_check; then
					echo 'Waiting for FIDO2 token'
					i=0
					while $fido2_token_check && [ \$i -lt $fido2_token_timeout ]; do
						usleep 1000000
						i=\$((i + 1))
					done
				fi
				EOF
			fi
			is_dm "$src" || echo "wait_blk_dev '$src'" | add_linuxrc
			verbose "+ cryptsetup ${keyfile:+-d $keyfile} open $crypttab_opt '$src' '$dst'"
			add_linuxrc <<-EOF
			debugshell

			cryptsetup_opt=""
			if [ "\$DEBUGINITRD" ]; then
				cryptsetup_opt="--debug"
			fi
			# cryptsetup can be called twice and in case on crypt on lvm only second
			# will succeed because there will be no src device in first cryptsetup call
			# this can be called multiple times, before lvm and after lvm.
			luksdev='$src'
			if [ \${luksdev##/dev/disk/by-uuid/} != \${luksdev} ]; then
			        src_uuid=\${luksdev##/dev/disk/by-uuid/}
			        while read x y z name; do
			                found_uuid=\$(cryptsetup \$cryptsetup_opt luksUUID /dev/\${name} 2>/dev/null)
			                if [ "\$found_uuid" = "\$src_uuid" ]; then
			                        luksdev=/dev/\$name
			                        break
			                fi
			        done < /proc/partitions
			fi

			if [ -e "\$luksdev" ]; then
				crypt_status=\$(cryptsetup \$cryptsetup_opt status '$dst')
				if [ "\${crypt_status%%is inactive*}" != "\$crypt_status" ]; then
					# is inactive
					cryptsetup \$cryptsetup_opt ${keyfile:+-d $keyfile} open $crypttab_opt "\$luksdev" '$dst' <&1
				fi
			fi

			debugshell
			EOF
		else
			die "$dst: only LUKS encryption supported"
		fi
	done < /etc/crypttab
}
