#!/bin/sh
# geninitrd mod: LVM
USE_LVM=${USE_LVM:-yes}

# if we should init LVM at boot
have_lvm=no

# LVM volumes that are needed to activate
# VG for root dev
# @internal
LVM_ROOTVG=""

# VG for suspend resume dev
LVM_SUSPENDVG=""

LVM_VGVOLUMES=""

# LVM version. Autodetected if not set.
# Values: 1|2
LVMTOOLSVERSION=

# LVM devices that should not be included in vgscan on initrd.
# @internal
lvm_ignore_devices=''

# setup geninitrd module
# @access	public
setup_mod_lvm() {
	lvm=$(find_tool $initrd_dir/lvm /sbin/initrd-lvm)

	if [ ! -x "$lvm" ]; then
		USE_LVM=no
	fi

	if is_yes "$USE_LVM" && [ -z "$LVMTOOLSVERSION" ]; then
		LVMTOOLSVERSION=$(LC_ALL=C $lvm vgchange --version 2>/dev/null | awk '/LVM version:/{if ($3 >= 2) print "2"}')
		if [ -z "$LVMTOOLSVERSION" ]; then
			die "Can't determine LVM tools version (is /sys mounted?). Please set LVMTOOLSVERSION and rerun $PROGRAM."
		fi
	fi
}

# return true if node is lvm node
# @param	string $node device node to be examined
# @access	public
is_lvm() {
	local node="$1"

	# LVM not wanted
	if is_no "$USE_LVM"; then
		return 1
	fi

	if [ ! -e "$node" ]; then
		die "check_lvm(): node $node doesn't exist!"
		return 1
	fi

	# block-major-58 is lvm1
	ls -lL "$node" 2> /dev/null | awk '{if (/^b/) { if ($5 == "58,") { exit 0; } else { exit 1; } } else { exit 1; }}'
	rc=$?

	if [ $rc = 0 ]; then
		# is lvm1
		return 0
	fi

	$lvm lvdisplay "$node" > /dev/null 2>&1
	rc=$?
	if [ $rc -gt 127 ]; then
		# lvdisplay terminated by signal! most likely it segfaulted.
		die "Unexpected exit from '$lvm lvdisplay $node': $rc - are your lvm tools broken?"
	fi

	return $rc
}

# find modules for $devpath
# @param	$devpath	device to be examined
# @access	public
find_modules_lvm() {
	local devpath="$1"

	verbose "LVM: $devpath is LVM node"

	local vg=$(find_lvm_vg "$devpath")
	verbose "LVM VG for $devpath: $vg"
	LVM_VGVOLUMES=$(echo $LVM_VGVOLUMES $vg | tr ' ' '\n' | sort -u)

	local pv=$(find_lvm_pv "$vg")
	verbose "LVM PV for $vg: $pv"
	PVDEVICES=$(echo $PVDEVICES $pv | tr ' ' '\n' | sort -u)

	if [ -n "$PVDEVICES" ]; then
		for device in $PVDEVICES; do
			find_modules_for_devpath $device
		done
	else
		die "I wasn't able to find PV. You can try to set PVDEVICES in /etc/sysconfig/geninitrd."
	fi

	if [ "$LVMTOOLSVERSION" = "2" ]; then
		find_module "-dm-mod"
	elif [ "$LVMTOOLSVERSION" = "1" ]; then
		find_module "-lvm"
		find_module "-lvm-mod"
	else
		die "LVM version $LVMTOOLSVERSION is not supported."
	fi

	verbose "LVM v$LVMTOOLSVERSION enabled"
	have_lvm=yes
}


# generate initrd fragment for lvm
# @access	public
initrd_gen_lvm() {
	if ! is_yes "$have_lvm"; then
		return
	fi

	verbose "Adding LVM support to initrd"
	inst_d /tmp /newroot
	inst_exec $lvm /bin/lvm.static

	# always make /dev on tmpfs for LVM2
	if [ "$LVMTOOLSVERSION" = "2" ]; then
		mount_dev
	fi

	if ! is_yes "$dev_mounted"; then
		inst_d /dev/mapper
		mknod -m 600 $DESTDIR/dev/mapper/control c 10 63
		for device in $PVDEVICES; do
			# if LVM on RAID then device might be copied already in gen_md
			[ -e "$DESTDIR/dev/$(basename $device)" ] && continue
			inst $device /dev
		done
	fi

	mount_sys
	if [ "$LVMTOOLSVERSION" = "1" ]; then
		add_linuxrc <<-EOF
			lvm vgscan -T
			for vg in $LVM_VGVOLUMES; do
				lvm vgchange -T -a y $vg
			done
		EOF
	else
		cat <<-EOF > "$DESTDIR/etc/lvm.conf"
		global {
			locking_type = 0
			locking_dir = "/tmp"
		}
		devices {
			sysfs_scan=0
		EOF
		if is_yes "$have_md"; then
			echo "	md_component_detection=1" >> "$DESTDIR/etc/lvm.conf"
		fi
		if is_yes "$have_dmraid" || is_yes "$have_multipath"; then
			echo '	types = [ "device-mapper", 254 ]' >> "$DESTDIR/etc/lvm.conf"
		fi
		if [ "$lvm_ignore_devices" ]; then
			# TODO: think of merging with lvm dumpconfig output
			echo '	filter = [' >> "$DESTDIR/etc/lvm.conf"
			local dev
			for dev in $lvm_ignore_devices; do
				verbose "LVM v2: ignore device $dev"
				printf '  "r|^%s.*|",\n' $dev
			done >> "$DESTDIR/etc/lvm.conf"
			echo ']' >> "$DESTDIR/etc/lvm.conf"
		fi
		# XXX filter= must be on one line!
		$lvm dumpconfig | awk '/filter=/' >> "$DESTDIR/etc/lvm.conf"
		echo "}" >> "$DESTDIR/etc/lvm.conf"

		echo "cat /etc/lvm.conf > /tmp/lvm.conf" | add_linuxrc

		initrd_gen_devices

		add_linuxrc <<-EOF
			ROOTDEV=$rootdev
			LVM_ROOTVG="$LVM_VGVOLUMES"
			LVM_SUSPENDVG="$LVM_SUSPENDVG"
		EOF

		# need awk for "s/--/-/g" subst when parsing kernel root commandline
		busybox_applet awk
		add_linuxrc <<-'EOF'
			# parse rootdev from kernel commandline if it begins with /
			case "$ROOT" in
				/*)

				# rewrite:
			 	# /dev/mapper/sys-rootfs -> /dev/sys/rootfs
				# /dev/mapper/blodnatt-blah--bleh -> /dev/blodnatt/blah-bleh
				# /dev/mapper/vg--meaw-root -> /dev/vg-meaw/root
				case "$ROOT" in
					/dev/mapper/*-*)
						# change "--" to / (as "/" is impossible in LV name)
						local dev=$(awk -vdev="${ROOT#/dev/mapper/}" 'BEGIN{gsub(/--/, "/", dev); print dev}')
						local VG=$(awk -vdev="$dev" 'BEGIN{split(dev, v, "-"); gsub("/", "-", v[1]); print v[1]}')
						local LV=$(awk -vdev="$dev" 'BEGIN{split(dev, v, "-"); gsub("/", "-", v[2]); print v[2]}')
						ROOT=/dev/$VG/$LV
					;;
				esac

				if [ "$ROOT" != "$ROOTDEV" ]; then
					ROOTDEV=$ROOT

					echo "LVM: Using 'root=$ROOTDEV' from kernel commandline"
					local tmp=${ROOTDEV#/dev/}
					if [ "$tmp" != "$ROOTDEV" ]; then
						LVM_ROOTVG=${tmp%/*}
						echo "LVM: Using Volume Group '$LVM_ROOTVG' for rootfs"
					fi
				fi
				;;
			esac

			# skip duplicate VG
			if [ "$LVM_SUSPENDVG" = "$LVM_ROOTVG" ]; then
				LVM_VGVOLUMES="$LVM_ROOTVG"
			else
				LVM_VGVOLUMES="$LVM_SUSPENDVG $LVM_ROOTVG"
			fi

			# disable noise from LVM accessing devices that aren't ready.
			read printk < /proc/sys/kernel/printk
			if [ ! "$DEBUGINITRD" ]; then
				echo 0 > /proc/sys/kernel/printk
			fi

			export LVM_SYSTEM_DIR=/tmp

			: 'Scanning for Volume Groups'
			lvm.static vgscan --mknodes --ignorelockingfailure

			: 'Activating Volume Groups'
			for vol in $LVM_VGVOLUMES; do
				lvm.static vgchange --ignorelockingfailure -a y $vol --noudevsync
			done

			: 'Extra call to make device nodes for non lvm2-initrd (dynamic lvm2)'
			lvm.static vgmknodes --ignorelockingfailure

			echo "$printk" > /proc/sys/kernel/printk

			# Find out major/minor
			attrs="$(lvm.static lvdisplay --ignorelockingfailure -c $ROOTDEV)"
			if [ "$attrs" ]; then
				majmin="${attrs#*$ROOTDEV*:*:*:*:*:*:*:*:*:*:*:*}"
				if [ "$majmin" != "$attrs" ]; then
					major="${majmin%:*}"
					minor="${majmin#*:}"
				fi
			fi

			if [ "$major" -a "$minor" ]; then
				# Pass it to kernel
				echo $((256 * $major + $minor)) > /proc/sys/kernel/real-root-dev
			fi

			debugshell
			unset LVM_SYSTEM_DIR
		EOF
	fi
}


# PRIVATE METHODS

# find PV names from given VG name
# @param	string $vg Volume Group name
# @access	private
find_lvm_pv() {
	local vg="$1"

	local pv=$($lvm vgs --noheadings -o pv_name  "$vg")
	echo $pv
}

# find VG name from given devnode
# @param	string $devnode device node to be examined
# @access	private
find_lvm_vg() {
	local devnode="$1"

	local vg=$($lvm lvs --noheadings -o vg_name  "$devnode")
	echo $vg
}
