# -*- coding: utf-8 -*-
"""Test suite for prance.util.fs ."""

__author__ = 'Jens Finkhaeuser'
__copyright__ = 'Copyright (c) 2016-2018 Jens Finkhaeuser'
__license__ = 'MIT +no-false-attribs'
__all__ = ()


import os
import sys

import pytest

from prance.util import fs

from . import sandbox


def test_canonical():
  testname = 'tests/specs/symlink_test'
  if sys.platform != "win32":
    res = fs.canonical_filename(testname)
    expected = os.path.join(os.getcwd(), 'tests/specs/with_externals.yaml')
    assert res == expected

def test_to_posix_rel():
  test = "tests/specs/with_externals.yaml"
  assert fs.to_posix(os.path.normpath(test)) == test

def test_to_posix_abs():
  if sys.platform == "win32":
    test = "c:\\windows\\notepad.exe"
    expected = "/c:/windows/notepad.exe"
  else:
    test = "/etc/passwd"
    expected = test
  assert fs.to_posix(test) == expected

def test_from_posix_rel():
  test = "tests/specs/with_externals.yaml"
  assert fs.from_posix(test) == os.path.normpath(test)

def test_from_posix_abs():
  if sys.platform == "win32":
    test = "/c:/windows/notepad.exe"
    expected = "c:\\windows\\notepad.exe"
  else:
    test = "/etc/passwd"
    expected = test
  assert fs.from_posix(test) == expected

def test_abspath_basics():
  testname = os.path.normpath('tests/specs/with_externals.yaml')
  res = fs.abspath(testname)
  expected = fs.to_posix(os.path.join(os.getcwd(), testname))
  assert res == expected

def test_abspath_relative():
  testname = 'error.json'
  relative = os.path.join(os.getcwd(), 'tests/specs/with_externals.yaml')
  res = fs.abspath(testname, relative)
  expected = fs.to_posix(os.path.join(os.getcwd(), 'tests', 'specs', testname))
  assert res == expected


def test_abspath_relative_dir():
  testname = 'error.json'
  relative = os.path.join(os.getcwd(), 'tests', 'specs')
  res = fs.abspath(testname, relative)
  expected = fs.to_posix(os.path.join(os.getcwd(), 'tests', 'specs', testname))
  assert res == expected


def test_detect_encoding():
  # Quick detection should yield utf-8 for the petstore file.
  assert fs.detect_encoding('tests/specs/petstore.yaml') == 'utf-8'

  # Really, it should be detected as ISO-8859-1 as a superset of ASCII
  assert fs.detect_encoding('tests/specs/petstore.yaml',
                            default_to_utf8 = False) == 'iso-8859-1'

  # Deep inspection should yield UTF-8 again.
  assert fs.detect_encoding('tests/specs/petstore.yaml',
                            default_to_utf8 = False,
                            read_all = True) == 'utf-8'

  # The UTF-8 file with BOM should be detected properly
  assert fs.detect_encoding('tests/specs/utf8bom.yaml') == 'utf-8-sig'


def test_load_nobom():
  contents = fs.read_file('tests/specs/petstore.yaml')
  assert contents.index(u'Swagger Petstore') >= 0, 'File reading failed!'


def test_load_utf8bom():
  contents = fs.read_file('tests/specs/utf8bom.yaml')
  assert contents.index(u'söme välüe') >= 0, 'UTF-8 BOM handling failed!'


def test_load_utf8bom_override():
  with pytest.raises(UnicodeDecodeError):
    fs.read_file('tests/specs/utf8bom.yaml', 'ascii')


def test_write_file(tmpdir):
  with sandbox.sandbox(tmpdir):
    test_text = u'söme täxt'
    fs.write_file('test.out', test_text)

    # File must have been written
    files = [f for f in os.listdir('.') if os.path.isfile(f)]
    assert 'test.out' in files

    # File contents must work
    contents = fs.read_file('test.out')
    assert test_text == contents


def test_write_file_bom(tmpdir):
  with sandbox.sandbox(tmpdir):
    test_text = u'söme täxt'
    fs.write_file('test.out', test_text, 'utf-8-sig')

    # File must have been written
    files = [f for f in os.listdir('.') if os.path.isfile(f)]
    assert 'test.out' in files

    # Encoding must match the one we've given
    encoding = fs.detect_encoding('test.out')
    assert encoding == 'utf-8-sig'

    # File contents must work
    contents = fs.read_file('test.out')
    assert test_text == contents


def test_valid_pathname():
  # A URL should not be valid
  from prance.util.fs import is_pathname_valid
  assert False == is_pathname_valid('\x00o.bar.org')

  # However, the current path should be.
  import os
  assert True == is_pathname_valid(os.getcwd())

  # Can't put non-strings into this function
  assert True == is_pathname_valid('foo')
  assert False == is_pathname_valid(123)

  # Can't accept too long components
  assert False == is_pathname_valid('a'*256)

