/*
 * Copyright (c) 2017-2025, Arm Limited and Contributors. All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdbool.h>

#include <arch.h>
#include <arch_helpers.h>
#include <lib/cassert.h>
#include <lib/el3_runtime/pubsub.h>
#include <lib/extensions/sve.h>

CASSERT(SVE_VECTOR_LEN <= 2048, assert_sve_vl_too_long);
CASSERT(SVE_VECTOR_LEN >= 128, assert_sve_vl_too_short);
CASSERT((SVE_VECTOR_LEN % 128) == 0, assert_sve_vl_granule);

/*
 * Converts SVE vector size restriction in bytes to LEN according to ZCR_EL3 documentation.
 * VECTOR_SIZE = (LEN+1) * 128
 */
#define CONVERT_SVE_LENGTH(x)	(((x / 128) - 1))

void sve_init_el3(void)
{
	/* Restrict maximum SVE vector length (SVE_VECTOR_LEN+1) * 128. */
	write_zcr_el3(ZCR_EL3_LEN_MASK & CONVERT_SVE_LENGTH(SVE_VECTOR_LEN));
}

void sve_enable_per_world(per_world_context_t *per_world_ctx)
{
	u_register_t cptr_el3;

	/* Enable access to SVE functionality for all ELs. */
	cptr_el3 = per_world_ctx->ctx_cptr_el3;
	cptr_el3 = (cptr_el3 | CPTR_EZ_BIT);
	per_world_ctx->ctx_cptr_el3 = cptr_el3;
}

void sve_init_el2_unused(void)
{
	u_register_t reg;

	/*
	 * CPTR_EL2.TZ: Set to zero so that no SVE instruction execution is
	 *  trapped.
	 *
	 * CPTR_EL2.ZEN: Set to 0b11 so that no SVE instruction execution is
	 *  trapped.
	 */
	reg = read_cptr_el2();
	reg &= ~(CPTR_EL2_TZ_BIT);
	reg |= ULL(3) << CPTR_EL2_ZEN_SHIFT;
	write_cptr_el2(reg);
}

void sve_disable_per_world(per_world_context_t *per_world_ctx)
{
	u_register_t reg;

	/* Disable SVE and FPU since they share registers. */
	reg = per_world_ctx->ctx_cptr_el3;
	reg &= ~CPTR_EZ_BIT;	/* Trap SVE */
	per_world_ctx->ctx_cptr_el3 = reg;
}
