--
--  mkdemo.sql
--
--  $Id$
--
--  Creates a demo database
--
--  This file is part of the OpenLink Software Virtuoso Open-Source (VOS)
--  project.
--
--  Copyright (C) 1998-2025 OpenLink Software
--
--  This project is free software; you can redistribute it and/or modify it
--  under the terms of the GNU General Public License as published by the
--  Free Software Foundation; only version 2 of the License, dated June 1991.
--
--  This program is distributed in the hope that it will be useful, but
--  WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
--  General Public License for more details.
--
--  You should have received a copy of the GNU General Public License along
--  with this program; if not, write to the Free Software Foundation, Inc.,
--  51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
--
--

--ECHO BOTH "Loading Demo Database\n";

-- DROP TABLE Order_Details;
-- DROP TABLE Orders;
-- DROP TABLE Employees;
-- DROP TABLE Customers;
-- DROP TABLE Products;
-- DROP TABLE Suppliers;
-- DROP TABLE Shippers;
-- DROP TABLE Categories;

create procedure DB.DBA.exec_no_error (in expr varchar) {
  declare state, message, meta, result any;
  exec(expr, state, message, vector(), 0, meta, result);
}
;

create procedure DB.DBA.get_blob_from_dav(in filename varchar) returns any
{
        declare _content any;
        select blob_to_string (RES_CONTENT) into _content from WS.WS.SYS_DAV_RES where RES_FULL_PATH = filename;
        return _content;
}
;

create procedure DB.DBA.GET_PWD_FOR_VAD (in user_name varchar) returns varchar
{
  declare pwd varchar;
  pwd := pwd_magic_calc(user_name, (select U_PWD from WS.WS.SYS_DAV_USER where U_NAME=user_name), 1);
  if (pwd is null)
    pwd := user_name;
  return pwd;
}
;

create procedure ensure_demo_user ()
{
  if (exists (select 1 from SYS_USERS where U_NAME = 'demo'))
    return;
  DB.DBA.USER_CREATE ('demo', 'demo', vector ('FULL_NAME', 'Demo account', 'DAV_ENABLE', 1, 'LOGIN_QUALIFIER', 'Demo'));
  update DB.DBA.SYS_USERS set U_GROUP = http_admin_gid () where U_NAME = 'demo';
  DB.DBA.DAV_HOME_DIR_CREATE ('demo');
  update DB.DBA.SYS_USERS set U_GROUP = U_ID where U_NAME = 'demo';
};

ensure_demo_user ();

DB.DBA.exec_no_error('drop procedure ensure_demo_user');

-- load dropdemo.sql;

USE Demo;

DB.DBA.exec_no_error('CREATE TABLE demo.Categories(
  CategoryID INTEGER,
  CategoryName VARCHAR(15),
  Description LONG VARCHAR,
  Picture LONG VARBINARY,
  PRIMARY KEY (CategoryID))');

DB.DBA.exec_no_error('CREATE INDEX CategoryName ON demo.Categories (CategoryName)');

DB.DBA.exec_no_error('CREATE TABLE demo.Shippers(
  ShipperID INTEGER,
  CompanyName VARCHAR(40),
  Phone VARCHAR(24),
  PRIMARY KEY (ShipperID))');

DB.DBA.exec_no_error('CREATE TABLE demo.Suppliers(
  SupplierID INTEGER,
  CompanyName VARCHAR(40),
  ContactName VARCHAR(30),
  ContactTitle VARCHAR(30),
  Address VARCHAR(60),
  City VARCHAR(15),
  Region VARCHAR(15),
  PostalCode VARCHAR(10),
  Country VARCHAR(15),
  CountryCode VARCHAR (2),
  Phone VARCHAR(24),
  Fax VARCHAR(24),
  HomePage LONG VARCHAR,
  PRIMARY KEY (SupplierID))');

DB.DBA.exec_no_error('CREATE INDEX CompanyName1 ON demo.Suppliers (CompanyName)');
DB.DBA.exec_no_error('CREATE INDEX PostalCode1 ON demo.Suppliers (PostalCode)');

DB.DBA.exec_no_error('CREATE TABLE demo.Products(
  ProductID INTEGER,
  ProductName VARCHAR(40),
  SupplierID INTEGER,
  CategoryID INTEGER,
  QuantityPerUnit VARCHAR(20),
  UnitPrice DOUBLE PRECISION,
  UnitsInStock SMALLINT,
  UnitsOnOrder SMALLINT,
  ReorderLevel SMALLINT,
  Discontinued SMALLINT,
  PRIMARY KEY (ProductID),
  FOREIGN KEY (SupplierID) REFERENCES Suppliers,
  FOREIGN KEY (CategoryID) REFERENCES Categories)');

DB.DBA.exec_no_error('CREATE INDEX CategoryID ON demo.Products (CategoryID)');
DB.DBA.exec_no_error('CREATE INDEX ProductName ON demo.Products (ProductName)');
DB.DBA.exec_no_error('CREATE INDEX SupplierID ON demo.Products (SupplierID)');

DB.DBA.exec_no_error('CREATE TABLE demo.Customers(
  CustomerID VARCHAR(5),
  CompanyName VARCHAR(40),
  ContactName VARCHAR(30),
  ContactTitle VARCHAR(30),
  Address VARCHAR(60),
  City VARCHAR(15),
  Region VARCHAR(15),
  PostalCode VARCHAR(10),
  Country VARCHAR(15),
  CountryCode VARCHAR (2),
  Phone VARCHAR(24),
  Fax VARCHAR(24),
  PRIMARY KEY (CustomerID))');

DB.DBA.exec_no_error('CREATE INDEX City ON demo.Customers (City)');
DB.DBA.exec_no_error('CREATE INDEX CompanyName2 ON demo.Customers (CompanyName)');
DB.DBA.exec_no_error('CREATE INDEX PostalCode2 ON demo.Customers (PostalCode)');
DB.DBA.exec_no_error('CREATE INDEX Region ON demo.Customers (Region)');

DB.DBA.exec_no_error('CREATE TABLE demo.Employees(
  EmployeeID INTEGER,
  LastName VARCHAR(20),
  FirstName VARCHAR(10),
  Title VARCHAR(30),
  TitleOfCourtesy VARCHAR(25),
  BirthDate DATE,
  HireDate DATE,
  Address VARCHAR(60),
  City VARCHAR(15),
  Region VARCHAR(15),
  PostalCode VARCHAR(10),
  Country VARCHAR(15),
  CountryCode VARCHAR (2),
  HomePhone VARCHAR(24),
  Extension VARCHAR(4),
  Photo LONG VARBINARY,
  Notes LONG VARCHAR,
  ReportsTo INTEGER,
  PRIMARY KEY (EmployeeID))');

DB.DBA.exec_no_error('CREATE INDEX LastName ON demo.Employees (LastName)');
DB.DBA.exec_no_error('CREATE INDEX PostalCode3 ON demo.Employees (PostalCode)');

DB.DBA.exec_no_error('CREATE TABLE demo.Orders(
  OrderID INTEGER,
  CustomerID VARCHAR(5),
  EmployeeID INTEGER,
  OrderDate DATETIME,
  RequiredDate DATE,
  ShippedDate DATETIME,
  ShipVia INTEGER,
  Freight DOUBLE PRECISION,
  ShipName VARCHAR(40),
  ShipAddress VARCHAR(60),
  ShipCity VARCHAR(15),
  ShipRegion VARCHAR(15),
  ShipPostalCode VARCHAR(10),
  ShipCountry VARCHAR(15),
  ShipCountryCode VARCHAR (2),
  PRIMARY KEY (OrderID),
  FOREIGN KEY (CustomerID) REFERENCES Customers,
  FOREIGN KEY (ShipVia) REFERENCES Shippers,
  FOREIGN KEY (EmployeeID) REFERENCES Employees)');

DB.DBA.exec_no_error('CREATE INDEX CustomerID ON demo.Orders (CustomerID)');
DB.DBA.exec_no_error('CREATE INDEX EmployeeID ON demo.Orders (EmployeeID)');
DB.DBA.exec_no_error('CREATE INDEX OrderDate ON demo.Orders (OrderDate)');
DB.DBA.exec_no_error('CREATE INDEX ShippedDate ON demo.Orders (ShippedDate)');

DB.DBA.exec_no_error('CREATE TABLE demo.Order_Details(
  OrderID INTEGER,
  ProductID INTEGER,
  UnitPrice DOUBLE PRECISION,
  Quantity SMALLINT,
  Discount REAL,
  PRIMARY KEY (OrderID, ProductID),
  FOREIGN KEY (OrderID) REFERENCES Orders,
  FOREIGN KEY (ProductID) REFERENCES Products)');

DB.DBA.exec_no_error('CREATE INDEX OrderID ON demo.Order_Details (OrderID)');
DB.DBA.exec_no_error('CREATE INDEX ProductID ON demo.Order_Details (ProductID)');

--
--
--

DB.DBA.exec_no_error('DELETE FROM Categories');

INSERT soft Categories(CategoryID,CategoryName,Description,Picture) VALUES(1, 'Beverages', 'Soft drinks, coffees, teas, beers, and ales', DB.DBA.get_blob_from_dav('/DAV/VAD/demo/sql/CAT1'));
INSERT soft Categories(CategoryID,CategoryName,Description,Picture) VALUES(2, 'Condiments', 'Sweet and savory sauces, relishes, spreads, and seasonings', DB.DBA.get_blob_from_dav('/DAV/VAD/demo/sql/CAT3'));
INSERT soft Categories(CategoryID,CategoryName,Description,Picture) VALUES(3, 'Confections', 'Desserts, candies, and sweet breads', DB.DBA.get_blob_from_dav('/DAV/VAD/demo/sql/CAT4'));
INSERT soft Categories(CategoryID,CategoryName,Description,Picture) VALUES(4, 'Dairy Products', 'Cheeses', DB.DBA.get_blob_from_dav('/DAV/VAD/demo/sql/CAT5'));
INSERT soft Categories(CategoryID,CategoryName,Description,Picture) VALUES(5, 'Grains/Cereals', 'Breads, crackers, pasta, and cereal', DB.DBA.get_blob_from_dav('/DAV/VAD/demo/sql/CAT6'));
INSERT soft Categories(CategoryID,CategoryName,Description,Picture) VALUES(6, 'Meat/Poultry', 'Prepared meats', DB.DBA.get_blob_from_dav('/DAV/VAD/demo/sql/CAT7'));
INSERT soft Categories(CategoryID,CategoryName,Description,Picture) VALUES(7, 'Produce', 'Dried fruit and bean curd', DB.DBA.get_blob_from_dav('/DAV/VAD/demo/sql/CAT8'));
INSERT soft Categories(CategoryID,CategoryName,Description,Picture) VALUES(8, 'Seafood', 'Seaweed and fish', DB.DBA.get_blob_from_dav('/DAV/VAD/demo/sql/CAT1'));

select count(*) from Categories;
--ECHO BOTH $IF $EQU $LAST[1] 8 "PASSED" "***FAILED";
--SET ARGV[$LIF] $+ $ARGV[$LIF] 1;
--ECHO BOTH ": " $LAST[1] " Categories loaded\n";

--
--
--

DB.DBA.exec_no_error('DELETE FROM Shippers');

INSERT soft Shippers(ShipperID,CompanyName,Phone) VALUES(1, 'Speedy Express', '(503) 555-9831');
INSERT soft Shippers(ShipperID,CompanyName,Phone) VALUES(2, 'United Package', '(503) 555-3199');
INSERT soft Shippers(ShipperID,CompanyName,Phone) VALUES(3, 'Federal Shipping', '(503) 555-9931');

select count(*) from Shippers;
--ECHO BOTH $IF $EQU $LAST[1] 3 "PASSED" "***FAILED";
--SET ARGV[$LIF] $+ $ARGV[$LIF] 1;
--ECHO BOTH ": " $LAST[1] " Shippers loaded\n";

--
--
--

DB.DBA.exec_no_error('DELETE FROM Suppliers');

INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(1,'Exotic Liquids','Charlotte Cooper','Purchasing Manager','49 Gilbert St.','London',NULL,'EC1 4SD','United Kingdom','(171) 555-2222',NULL,'');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(2,'New Orleans Cajun Delights','Shelley Burke','Order Administrator','P.O. Box 78934','New Orleans','LA','70117','United States','(100) 555-4822',NULL,'#CAJUN.HTM#');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(3,'Grandma Kelly\47s Homestead','Regina Murphy','Sales Representative','707 Oxford Rd.','Ann Arbor','MI','48104','United States','(313) 555-5735','(313) 555-3349','');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(4,'Tokyo Traders','Yoshi Nagase','Marketing Manager','9-8 Sekimai\15\12Musashino-shi','Tokyo',NULL,'100','Japan','(03) 3555-5011',NULL,'');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(5,'Cooperativa de Quesos \47Las Cabras\47','Antonio del Valle Saavedra ','Export Administrator','Calle del Rosal 4','Oviedo','Asturias','33007','Spain','(98) 598 76 54',NULL,'');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(6,'Mayumi\47s','Mayumi Ohno','Marketing Representative','92 Setsuko\15\12Chuo-ku','Osaka',NULL,'545','Japan','(06) 431-7877',NULL,'Mayumi\47s (on the World Wide Web)#http://www.microsoft.com/accessdev/sampleapps/mayumi.htm#');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(7,'Pavlova, Ltd.','Ian Devling','Marketing Manager','74 Rose St.\15\12Moonie Ponds','Melbourne','Victoria','3058','Australia','(03) 444-2343','(03) 444-6588','');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(8,'Specialty Biscuits, Ltd.','Peter Wilson','Sales Representative','29 King\47s Way','Manchester',NULL,'M14 GSD','United Kingdom','(161) 555-4448',NULL,'');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(9,'PB Knckebrd AB','Lars Peterson','Sales Agent','Kaloadagatan 13','Gteborg',NULL,'S-345 67','Sweden','031-987 65 43','031-987 65 91','');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(10,'Refrescos Americanas LTDA','Carlos Diaz','Marketing Manager','Av. das Americanas 12.890','So Paulo',NULL,'5442','Brazil','(11) 555 4640',NULL,'');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(11,'Heli Swaren GmbH & Co. KG','Petra Winkler','Sales Manager','Tiergartenstrae 5','Berlin',NULL,'10785','Germany','(010) 9984510',NULL,'');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(12,'Plutzer Lebensmittelgromrkte AG','Martin Bein','International Marketing Mgr.','Bogenallee 51','Frankfurt',NULL,'60439','Germany','(069) 992755',NULL,'Plutzer (on the World Wide Web)#http://www.microsoft.com/accessdev/sampleapps/plutzer.htm#');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(13,'Nord-Ost-Fisch Handelsgesellschaft mbH','Sven Petersen','Coordinator Foreign Markets','Frahmredder 112a','Cuxhaven',NULL,'27478','Germany','(04721) 8713','(04721) 8714','');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(14,'Formaggi Fortini s.r.l.','Elio Rossi','Sales Representative','Viale Dante, 75','Ravenna',NULL,'48100','Italy','(0544) 60323','(0544) 60603','#FORMAGGI.HTM#');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(15,'Norske Meierier','Beate Vileid','Marketing Manager','Hatlevegen 5','Sandvika',NULL,'1320','Norway','(0)2-953010',NULL,'');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(16,'Bigfoot Breweries','Cheryl Saylor','Regional Account Rep.','3400 - 8th Avenue\15\12Suite 210','Bend','OR','97101','United States','(503) 555-9931',NULL,'');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(17,'Svensk Sjfda AB','Michael Bjrn','Sales Representative','Brovallavgen 231','Stockholm',NULL,'S-123 45','Sweden','08-123 45 67',NULL,'');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(18,'Aux joyeux ecclsiastiques','Guylne Nodier','Sales Manager','203, Rue des Francs-Bourgeois','Paris',NULL,'75004','France','(1) 03.83.00.68','(1) 03.83.00.62','');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(19,'New England Seafood Cannery','Robb Merchant','Wholesale Account Agent','Order Processing Dept.\15\122100 Paul Revere Blvd.','Boston','MA','02134','United States','(617) 555-3267','(617) 555-3389','');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(20,'Leka Trading','Chandra Leka','Owner','471 Serangoon Loop, Suite #402','Singapore',NULL,'0512','Singapore','555-8787',NULL,'');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(21,'Lyngbysild','Niels Petersen','Sales Manager','Lyngbysild\15\12Fiskebakken 10','Lyngby',NULL,'2800','Denmark','43844108','43844115','');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(22,'Zaanse Snoepfabriek','Dirk Luchte','Accounting Manager','Verkoop\15\12Rijnweg 22','Zaandam',NULL,'9999 ZZ','Netherlands','(12345) 1212','(12345) 1210','');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(23,'Karkki Oy','Anne Heikkonen','Product Manager','Valtakatu 12','Lappeenranta',NULL,'53120','Finland','(953) 10956',NULL,'');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(24,'G\47day, Mate','Wendy Mackenzie','Sales Representative','170 Prince Edward Parade\15\12Hunter\47s Hill','Sydney','NSW','2042','Australia','(02) 555-5914','(02) 555-4873','G\47day Mate (on the World Wide Web)#http://www.microsoft.com/accessdev/sampleapps/gdaymate.htm#');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(25,'Ma Maison','Jean-Guy Lauzon','Marketing Manager','2960 Rue St. Laurent','Montral','Qubec','H1J 1C3','Canada','(514) 555-9022',NULL,'');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(26,'Pasta Buttini s.r.l.','Giovanni Giudici','Order Administrator','Via dei Gelsomini, 153','Salerno',NULL,'84100','Italy','(089) 6547665','(089) 6547667','');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(27,'Escargots Nouveaux','Marie Delamare','Sales Manager','22, rue H. Voiron','Montceau',NULL,'71300','France','85.57.00.07',NULL,'');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(28,'Gai pturage','Eliane Noz','Sales Representative','Bat. B\15\123, rue des Alpes','Annecy',NULL,'74000','France','38.76.98.06','38.76.98.58','');
INSERT soft Suppliers(SupplierID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax,HomePage) VALUES(29,'Forts d\47rables','Chantal Goulet','Accounting Manager','148 rue Chasseur','Ste-Hyacinthe','Qubec','J2S 7S8','Canada','(514) 555-2955','(514) 555-2921','');

select count(*) from Suppliers;
--ECHO BOTH $IF $EQU $LAST[1] 29 "PASSED" "***FAILED";
--SET ARGV[$LIF] $+ $ARGV[$LIF] 1;
--ECHO BOTH ": " $LAST[1] " Suppliers loaded\n";

--
--
--

DB.DBA.exec_no_error('DELETE FROM Products');

INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(1,'Chai',1,1,'10 boxes x 20 bags',18.000000,39,0,10,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(2,'Chang',1,1,'24 - 12 oz bottles',19.000000,17,40,25,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(3,'Aniseed Syrup',1,2,'12 - 550 ml bottles',10.000000,13,70,25,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(4,'Chef Anton\47s Cajun Seasoning',2,2,'48 - 6 oz jars',22.000000,53,0,0,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(5,'Chef Anton\47s Gumbo Mix',2,2,'36 boxes',21.350000,0,0,0,1);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(6,'Grandma\47s Boysenberry Spread',3,2,'12 - 8 oz jars',25.000000,120,0,25,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(7,'Uncle Bob\47s Organic Dried Pears',3,7,'12 - 1 lb pkgs.',30.000000,15,0,10,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(8,'Northwoods Cranberry Sauce',3,2,'12 - 12 oz jars',40.000000,6,0,0,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(9,'Mishi Kobe Niku',4,6,'18 - 500 g pkgs.',97.000000,29,0,0,1);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(10,'Ikura',4,8,'12 - 200 ml jars',31.000000,31,0,0,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(11,'Queso Cabrales',5,4,'1 kg pkg.',21.000000,22,30,30,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(12,'Queso Manchego La Pastora',5,4,'10 - 500 g pkgs.',38.000000,86,0,0,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(13,'Konbu',6,8,'2 kg box',6.000000,24,0,5,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(14,'Tofu',6,7,'40 - 100 g pkgs.',23.250000,35,0,0,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(15,'Genen Shouyu',6,2,'24 - 250 ml bottles',15.500000,39,0,5,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(16,'Pavlova',7,3,'32 - 500 g boxes',17.450000,29,0,10,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(17,'Alice Mutton',7,6,'20 - 1 kg tins',39.000000,0,0,0,1);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(18,'Carnarvon Tigers',7,8,'16 kg pkg.',62.500000,42,0,0,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(19,'Teatime Chocolate Biscuits',8,3,'10 boxes x 12 pieces',9.200000,25,0,5,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(20,'Sir Rodney\47s Marmalade',8,3,'30 gift boxes',81.000000,40,0,0,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(21,'Sir Rodney\47s Scones',8,3,'24 pkgs. x 4 pieces',10.000000,3,40,5,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(22,'Gustaf\47s Knckebrd',9,5,'24 - 500 g pkgs.',21.000000,104,0,25,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(23,'Tunnbrd',9,5,'12 - 250 g pkgs.',9.000000,61,0,25,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(24,'Guaran Fantstica',10,1,'12 - 355 ml cans',4.500000,20,0,0,1);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(25,'NuNuCa Nu-Nougat-Creme',11,3,'20 - 450 g glasses',14.000000,76,0,30,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(26,'Gumbr Gummibrchen',11,3,'100 - 250 g bags',31.230000,15,0,0,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(27,'Schoggi Schokolade',11,3,'100 - 100 g pieces',43.900000,49,0,30,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(28,'Rssle Sauerkraut',12,7,'25 - 825 g cans',45.600000,26,0,0,1);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(29,'Thringer Rostbratwurst',12,6,'50 bags x 30 sausgs.',123.790000,0,0,0,1);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(30,'Nord-Ost Matjeshering',13,8,'10 - 200 g glasses',25.890000,10,0,15,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(31,'Gorgonzola Telino',14,4,'12 - 100 g pkgs',12.500000,0,70,20,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(32,'Mascarpone Fabioli',14,4,'24 - 200 g pkgs.',32.000000,9,40,25,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(33,'Geitost',15,4,'500 g',2.500000,112,0,20,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(34,'Sasquatch Ale',16,1,'24 - 12 oz bottles',14.000000,111,0,15,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(35,'Steeleye Stout',16,1,'24 - 12 oz bottles',18.000000,20,0,15,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(36,'Inlagd Sill',17,8,'24 - 250 g  jars',19.000000,112,0,20,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(37,'Gravad lax',17,8,'12 - 500 g pkgs.',26.000000,11,50,25,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(38,'Cte de Blaye',18,1,'12 - 75 cl bottles',263.500000,17,0,15,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(39,'Chartreuse verte',18,1,'750 cc per bottle',18.000000,69,0,5,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(40,'Boston Crab Meat',19,8,'24 - 4 oz tins',18.400000,123,0,30,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(41,'Jack\47s New England Clam Chowder',19,8,'12 - 12 oz cans',9.650000,85,0,10,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(42,'Singaporean Hokkien Fried Mee',20,5,'32 - 1 kg pkgs.',14.000000,26,0,0,1);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(43,'Ipoh Coffee',20,1,'16 - 500 g tins',46.000000,17,10,25,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(44,'Gula Malacca',20,2,'20 - 2 kg bags',19.450000,27,0,15,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(45,'Rgede sild',21,8,'1k pkg.',9.500000,5,70,15,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(46,'Spegesild',21,8,'4 - 450 g glasses',12.000000,95,0,0,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(47,'Zaanse koeken',22,3,'10 - 4 oz boxes',9.500000,36,0,0,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(48,'Chocolade',22,3,'10 pkgs.',12.750000,15,70,25,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(49,'Maxilaku',23,3,'24 - 50 g pkgs.',20.000000,10,60,15,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(50,'Valkoinen suklaa',23,3,'12 - 100 g bars',16.250000,65,0,30,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(51,'Manjimup Dried Apples',24,7,'50 - 300 g pkgs.',53.000000,20,0,10,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(52,'Filo Mix',24,5,'16 - 2 kg boxes',7.000000,38,0,25,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(53,'Perth Pasties',24,6,'48 pieces',32.800000,0,0,0,1);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(54,'Tourtire',25,6,'16 pies',7.450000,21,0,10,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(55,'Pt chinois',25,6,'24 boxes x 2 pies',24.000000,115,0,20,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(56,'Gnocchi di nonna Alice',26,5,'24 - 250 g pkgs.',38.000000,21,10,30,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(57,'Ravioli Angelo',26,5,'24 - 250 g pkgs.',19.500000,36,0,20,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(58,'Escargots de Bourgogne',27,8,'24 pieces',13.250000,62,0,20,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(59,'Raclette Courdavault',28,4,'5 kg pkg.',55.000000,79,0,0,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(60,'Camembert Pierrot',28,4,'15 - 300 g rounds',34.000000,19,0,0,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(61,'Sirop d\47rable',29,2,'24 - 500 ml bottles',28.500000,113,0,25,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(62,'Tarte au sucre',29,3,'48 pies',49.300000,17,0,0,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(63,'Vegie-spread',7,2,'15 - 625 g jars',43.900000,24,0,5,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(64,'Wimmers gute Semmelkndel',12,5,'20 bags x 4 pieces',33.250000,22,80,30,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(65,'Louisiana Fiery Hot Pepper Sauce',2,2,'32 - 8 oz bottles',21.050000,76,0,0,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(66,'Louisiana Hot Spiced Okra',2,2,'24 - 8 oz jars',17.000000,4,100,20,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(67,'Laughing Lumberjack Lager',16,1,'24 - 12 oz bottles',14.000000,52,0,10,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(68,'Scottish Longbreads',8,3,'10 boxes x 8 pieces',12.500000,6,10,15,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(69,'Gudbrandsdalsost',15,4,'10 kg pkg.',36.000000,26,0,15,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(70,'Outback Lager',7,1,'24 - 355 ml bottles',15.000000,15,10,30,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(71,'Fltemysost',15,4,'10 - 500 g pkgs.',21.500000,26,0,0,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(72,'Mozzarella di Giovanni',14,4,'24 - 200 g pkgs.',34.800000,14,0,0,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(73,'Rd Kaviar',17,8,'24 - 150 g jars',15.000000,101,0,5,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(74,'Longlife Tofu',4,7,'5 kg pkg.',10.000000,4,20,5,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(75,'Rhnbru Klosterbier',12,1,'24 - 0.5 l bottles',7.750000,125,0,25,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(76,'Lakkalikri',23,1,'500 ml',18.000000,57,0,20,0);
INSERT soft Products(ProductID,ProductName,SupplierID,CategoryID,QuantityPerUnit,UnitPrice,UnitsInStock,UnitsOnOrder,ReorderLevel,Discontinued) VALUES(77,'Original Frankfurter grne Soe',12,2,'12 boxes',13.000000,32,0,15,0);

select count(*) from Products;
--ECHO BOTH $IF $EQU $LAST[1] 77 "PASSED" "***FAILED";
--SET ARGV[$LIF] $+ $ARGV[$LIF] 1;
--ECHO BOTH ": " $LAST[1] " Products loaded\n";

--
--
--

DB.DBA.exec_no_error('DELETE FROM Customers');

INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('ALFKI','Alfreds Futterkiste','Maria Anders','Sales Representative','Obere Str. 57','Berlin',NULL,'12209','Germany','030-0074321','030-0076545');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('ANATR','Ana Trujillo Emparedados y helados','Ana Trujillo','Owner','Avda. de la Constitucin 2222','Mxico D.F.',NULL,'05021','Mexico','(5) 555-4729','(5) 555-3745');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('ANTON','Antonio Moreno Taquera','Antonio Moreno','Owner','Mataderos  2312','Mxico D.F.',NULL,'05023','Mexico','(5) 555-3932',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('AROUT','Around the Horn','Thomas Hardy','Sales Representative','120 Hanover Sq.','London',NULL,'WA1 1DP','United Kingdom','(171) 555-7788','(171) 555-6750');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('BERGS','Berglunds snabbkp','Christina Berglund','Order Administrator','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden','0921-12 34 65','0921-12 34 67');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('BLAUS','Blauer See Delikatessen','Hanna Moos','Sales Representative','Forsterstr. 57','Mannheim',NULL,'68306','Germany','0621-08460','0621-08924');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('BLONP','Blondel pre et fils','Frdrique Citeaux','Marketing Manager','24, place Klber','Strasbourg',NULL,'67000','France','88.60.15.31','88.60.15.32');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('BOLID','Blido Comidas preparadas','Martn Sommer','Owner','C/ Araquil, 67','Madrid',NULL,'28023','Spain','(91) 555 22 82','(91) 555 91 99');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('BONAP','Bon app\47','Laurence Lebihan','Owner','12, rue des Bouchers','Marseille',NULL,'13008','France','91.24.45.40','91.24.45.41');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('BOTTM','Bottom-Dollar Markets','Elizabeth Lincoln','Accounting Manager','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada','(604) 555-4729','(604) 555-3745');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('BSBEV','B\47s Beverages','Victoria Ashworth','Sales Representative','Fauntleroy Circus','London',NULL,'EC2 5NT','United Kingdom','(171) 555-1212',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('CACTU','Cactus Comidas para llevar','Patricio Simpson','Sales Agent','Cerrito 333','Buenos Aires',NULL,'1010','Argentina','(1) 135-5555','(1) 135-4892');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('CENTC','Centro comercial Moctezuma','Francisco Chang','Marketing Manager','Sierras de Granada 9993','Mxico D.F.',NULL,'05022','Mexico','(5) 555-3392','(5) 555-7293');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('CHOPS','Chop-suey Chinese','Yang Wang','Owner','Hauptstr. 29','Bern',NULL,'3012','Switzerland','0452-076545',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('COMMI','Comrcio Mineiro','Pedro Afonso','Sales Associate','Av. dos Lusadas, 23','So Paulo','SP','05432-043','Brazil','(11) 555-7647',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('CONSH','Consolidated Holdings','Elizabeth Brown','Sales Representative','Berkeley Gardens\15\1212  Brewery ','London',NULL,'WX1 6LT','United Kingdom','(171) 555-2282','(171) 555-9199');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('DRACD','Drachenblut Delikatessen','Sven Ottlieb','Order Administrator','Walserweg 21','Aachen',NULL,'52066','Germany','0241-039123','0241-059428');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('DUMON','Du monde entier','Janine Labrune','Owner','67, rue des Cinquante Otages','Nantes',NULL,'44000','France','40.67.88.88','40.67.89.89');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('EASTC','Eastern Connection','Ann Devon','Sales Agent','35 King George','London',NULL,'WX3 6FW','United Kingdom','(171) 555-0297','(171) 555-3373');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('ERNSH','Ernst Handel','Roland Mendel','Sales Manager','Kirchgasse 6','Graz',NULL,'8010','Austria','7675-3425','7675-3426');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('FAMIA','Familia Arquibaldo','Aria Cruz','Marketing Assistant','Rua Ors, 92','So Paulo','SP','05442-030','Brazil','(11) 555-9857',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('FISSA','FISSA Fabrica Inter. Salchichas S.A.','Diego Roel','Accounting Manager','C/ Moralzarzal, 86','Madrid',NULL,'28034','Spain','(91) 555 94 44','(91) 555 55 93');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('FOLIG','Folies gourmandes','Martine Ranc','Assistant Sales Agent','184, chausse de Tournai','Lille',NULL,'59000','France','20.16.10.16','20.16.10.17');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('FOLKO','Folk och f HB','Maria Larsson','Owner','kergatan 24','Brcke',NULL,'S-844 67','Sweden','0695-34 67 21',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('FRANK','Frankenversand','Peter Franken','Marketing Manager','Berliner Platz 43','Mnchen',NULL,'80805','Germany','089-0877310','089-0877451');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('FRANR','France restauration','Carine Schmitt','Marketing Manager','54, rue Royale','Nantes',NULL,'44000','France','40.32.21.21','40.32.21.20');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('FRANS','Franchi S.p.A.','Paolo Accorti','Sales Representative','Via Monte Bianco 34','Torino',NULL,'10100','Italy','011-4988260','011-4988261');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('FURIB','Furia Bacalhau e Frutos do Mar','Lino Rodriguez ','Sales Manager','Jardim das rosas n. 32','Lisboa',NULL,'1675','Portugal','(1) 354-2534','(1) 354-2535');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('GALED','Galera del gastrnomo','Eduardo Saavedra','Marketing Manager','Rambla de Catalua, 23','Barcelona',NULL,'08022','Spain','(93) 203 4560','(93) 203 4561');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('GODOS','Godos Cocina Tpica','Jos Pedro Freyre','Sales Manager','C/ Romero, 33','Sevilla',NULL,'41101','Spain','(95) 555 82 82',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('GOURL','Gourmet Lanchonetes','Andr Fonseca','Sales Associate','Av. Brasil, 442','Campinas','SP','04876-786','Brazil','(11) 555-9482',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('GREAL','Great Lakes Food Market','Howard Snyder','Marketing Manager','2732 Baker Blvd.','Eugene','OR','97403','United States','(503) 555-7555',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('GROSR','GROSELLA-Restaurante','Manuel Pereira','Owner','5 Ave. Los Palos Grandes','Caracas','DF','1081','Venezuela','(2) 283-2951','(2) 283-3397');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('HANAR','Hanari Carnes','Mario Pontes','Accounting Manager','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil','(21) 555-0091','(21) 555-8765');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('HILAA','HILARIN-Abastos','Carlos Hernndez','Sales Representative','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela','(5) 555-1340','(5) 555-1948');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('HUNGC','Hungry Coyote Import Store','Yoshi Latimer','Sales Representative','City Center Plaza\15\12516 Main St.','Elgin','OR','97827','United States','(503) 555-6874','(503) 555-2376');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('HUNGO','Hungry Owl All-Night Grocers','Patricia McKenna','Sales Associate','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland','2967 542','2967 3333');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('ISLAT','Island Trading','Helen Bennett','Marketing Manager','Garden House\15\12Crowther Way','Cowes','Isle of Wight','PO31 7PJ','United Kingdom','(198) 555-8888',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('KOENE','Kniglich Essen','Philip Cramer','Sales Associate','Maubelstr. 90','Brandenburg',NULL,'14776','Germany','0555-09876',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('LACOR','La corne d\47abondance','Daniel Tonini','Sales Representative','67, avenue de l\47Europe','Versailles',NULL,'78000','France','30.59.84.10','30.59.85.11');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('LAMAI','La maison d\47Asie','Annette Roulet','Sales Manager','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France','61.77.61.10','61.77.61.11');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('LAUGB','Laughing Bacchus Wine Cellars','Yoshi Tannamuri','Marketing Assistant','1900 Oak St.','Vancouver','BC','V3F 2K1','Canada','(604) 555-3392','(604) 555-7293');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('LAZYK','Lazy K Kountry Store','John Steel','Marketing Manager','12 Orchestra Terrace','Walla Walla','WA','99362','United States','(509) 555-7969','(509) 555-6221');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('LEHMS','Lehmanns Marktstand','Renate Messner','Sales Representative','Magazinweg 7','Frankfurt a.M. ',NULL,'60528','Germany','069-0245984','069-0245874');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('LETSS','Let\47s Stop N Shop','Jaime Yorres','Owner','87 Polk St.\15\12Suite 5','San Francisco','CA','94117','United States','(415) 555-5938',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('LILAS','LILA-Supermercado','Carlos Gonzlez','Accounting Manager','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela','(9) 331-6954','(9) 331-7256');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('LINOD','LINO-Delicateses','Felipe Izquierdo','Owner','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela','(8) 34-56-12','(8) 34-93-93');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('LONEP','Lonesome Pine Restaurant','Fran Wilson','Sales Manager','89 Chiaroscuro Rd.','Portland','OR','97219','United States','(503) 555-9573','(503) 555-9646');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('MAGAA','Magazzini Alimentari Riuniti','Giovanni Rovelli','Marketing Manager','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy','035-640230','035-640231');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('MAISD','Maison Dewey','Catherine Dewey','Sales Agent','Rue Joseph-Bens 532','Bruxelles',NULL,'B-1180','Belgium','(02) 201 24 67','(02) 201 24 68');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('MEREP','Mre Paillarde','Jean Fresnire','Marketing Assistant','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada','(514) 555-8054','(514) 555-8055');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('MORGK','Morgenstern Gesundkost','Alexander Feuer','Marketing Assistant','Heerstr. 22','Leipzig',NULL,'04179','Germany','0342-023176',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('NORTS','North/South','Simon Crowther','Sales Associate','South House\15\12300 Queensbridge','London',NULL,'SW7 1RZ','United Kingdom','(171) 555-7733','(171) 555-2530');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('OCEAN','Ocano Atlntico Ltda.','Yvonne Moncada','Sales Agent','Ing. Gustavo Moncada 8585\15\12Piso 20-A','Buenos Aires',NULL,'1010','Argentina','(1) 135-5333','(1) 135-5535');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('OLDWO','Old World Delicatessen','Rene Phillips','Sales Representative','2743 Bering St.','Anchorage','AK','99508','United States','(907) 555-7584','(907) 555-2880');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('OTTIK','Ottilies Kseladen','Henriette Pfalzheim','Owner','Mehrheimerstr. 369','Kln',NULL,'50739','Germany','0221-0644327','0221-0765721');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('PARIS','Paris spcialits','Marie Bertrand','Owner','265, boulevard Charonne','Paris',NULL,'75012','France','(1) 42.34.22.66','(1) 42.34.22.77');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('PERIC','Pericles Comidas clsicas','Guillermo Fernndez','Sales Representative','Calle Dr. Jorge Cash 321','Mxico D.F.',NULL,'05033','Mexico','(5) 552-3745','(5) 545-3745');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('PICCO','Piccolo und mehr','Georg Pipps','Sales Manager','Geislweg 14','Salzburg',NULL,'5020','Austria','6562-9722','6562-9723');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('PRINI','Princesa Isabel Vinhos','Isabel de Castro','Sales Representative','Estrada da sade n. 58','Lisboa',NULL,'1756','Portugal','(1) 356-5634',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('QUEDE','Que Delcia','Bernardo Batista','Accounting Manager','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil','(21) 555-4252','(21) 555-4545');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('QUEEN','Queen Cozinha','Lcia Carvalho','Marketing Assistant','Alameda dos Canrios, 891','So Paulo','SP','05487-020','Brazil','(11) 555-1189',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('QUICK','QUICK-Stop','Horst Kloss','Accounting Manager','Taucherstrae 10','Cunewalde',NULL,'01307','Germany','0372-035188',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('RANCH','Rancho grande','Sergio Gutirrez','Sales Representative','Av. del Libertador 900','Buenos Aires',NULL,'1010','Argentina','(1) 123-5555','(1) 123-5556');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('RATTC','Rattlesnake Canyon Grocery','Paula Wilson','Assistant Sales Representative','2817 Milton Dr.','Albuquerque','NM','87110','United States','(505) 555-5939','(505) 555-3620');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('REGGC','Reggiani Caseifici','Maurizio Moroni','Sales Associate','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy','0522-556721','0522-556722');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('RICAR','Ricardo Adocicados','Janete Limeira','Assistant Sales Agent','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil','(21) 555-3412',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('RICSU','Richter Supermarkt','Michael Holz','Sales Manager','Grenzacherweg 237','Genve',NULL,'1203','Switzerland','0897-034214',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('ROMEY','Romero y tomillo','Alejandra Camino','Accounting Manager','Gran Va, 1','Madrid',NULL,'28001','Spain','(91) 745 6200','(91) 745 6210');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('SANTG','Sant Gourmet','Jonas Bergulfsen','Owner','Erling Skakkes gate 78','Stavern',NULL,'4110','Norway','07-98 92 35','07-98 92 47');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('SAVEA','Save-a-lot Markets','Jose Pavarotti','Sales Representative','187 Suffolk Ln.','Boise','ID','83720','United States','(208) 555-8097',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('SEVES','Seven Seas Imports','Hari Kumar','Sales Manager','90 Wadhurst Rd.','London',NULL,'OX15 4NB','United Kingdom','(171) 555-1717','(171) 555-5646');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('SIMOB','Simons bistro','Jytte Petersen','Owner','Vinbltet 34','Kbenhavn',NULL,'1734','Denmark','31 12 34 56','31 13 35 57');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('SPECD','Spcialits du monde','Dominique Perrier','Marketing Manager','25, rue Lauriston','Paris',NULL,'75016','France','(1) 47.55.60.10','(1) 47.55.60.20');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('SPLIR','Split Rail Beer & Ale','Art Braunschweiger','Sales Manager','P.O. Box 555','Lander','WY','82520','United States','(307) 555-4680','(307) 555-6525');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('SUPRD','Suprmes dlices','Pascale Cartrain','Accounting Manager','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium','(071) 23 67 22 20','(071) 23 67 22 21');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('THEBI','The Big Cheese','Liz Nixon','Marketing Manager','89 Jefferson Way\15\12Suite 2','Portland','OR','97201','United States','(503) 555-3612',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('THECR','The Cracker Box','Liu Wong','Marketing Assistant','55 Grizzly Peak Rd.','Butte','MT','59801','United States','(406) 555-5834','(406) 555-8083');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('TOMSP','Toms Spezialitten','Karin Josephs','Marketing Manager','Luisenstr. 48','Mnster',NULL,'44087','Germany','0251-031259','0251-035695');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('TORTU','Tortuga Restaurante','Miguel Angel Paolino','Owner','Avda. Azteca 123','Mxico D.F.',NULL,'05033','Mexico','(5) 555-2933',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('TRADH','Tradio Hipermercados','Anabela Domingues','Sales Representative','Av. Ins de Castro, 414','So Paulo','SP','05634-030','Brazil','(11) 555-2167','(11) 555-2168');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('TRAIH','Trail\47s Head Gourmet Provisioners','Helvetius Nagy','Sales Associate','722 DaVinci Blvd.','Kirkland','WA','98034','United States','(206) 555-8257','(206) 555-2174');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('VAFFE','Vaffeljernet','Palle Ibsen','Sales Manager','Smagslget 45','rhus',NULL,'8200','Denmark','86 21 32 43','86 22 33 44');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('VICTE','Victuailles en stock','Mary Saveley','Sales Agent','2, rue du Commerce','Lyon',NULL,'69004','France','78.32.54.86','78.32.54.87');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('VINET','Vins et alcools Chevalier','Paul Henriot','Accounting Manager','59 rue de l\47Abbaye','Reims',NULL,'51100','France','26.47.15.10','26.47.15.11');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('WANDK','Die Wandernde Kuh','Rita Mller','Sales Representative','Adenauerallee 900','Stuttgart',NULL,'70563','Germany','0711-020361','0711-035428');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('WARTH','Wartian Herkku','Pirkko Koskitalo','Accounting Manager','Torikatu 38','Oulu',NULL,'90110','Finland','981-443655','981-443655');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('WELLI','Wellington Importadora','Paula Parente','Sales Manager','Rua do Mercado, 12','Resende','SP','08737-363','Brazil','(14) 555-8122',NULL);
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('WHITC','White Clover Markets','Karl Jablonski','Owner','305 - 14th Ave. S.\15\12Suite 3B','Seattle','WA','98128','United States','(206) 555-4112','(206) 555-4115');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('WILMK','Wilman Kala','Matti Karttunen','Owner/Marketing Assistant','Keskuskatu 45','Helsinki',NULL,'21240','Finland','90-224 8858','90-224 8858');
INSERT soft Customers(CustomerID,CompanyName,ContactName,ContactTitle,Address,City,Region,PostalCode,Country,Phone,Fax) VALUES('WOLZA','Wolski  Zajazd','Zbyszek Piestrzeniewicz','Owner','ul. Filtrowa 68','Warszawa',NULL,'01-012','Poland','(26) 642-7012','(26) 642-7012');

select count(*) from Customers;
--ECHO BOTH $IF $EQU $LAST[1] 91 "PASSED" "***FAILED";
--SET ARGV[$LIF] $+ $ARGV[$LIF] 1;
--ECHO BOTH ": " $LAST[1] " Customers loaded\n";

--
--
--

DB.DBA.exec_no_error('DELETE FROM Employees');

INSERT soft Employees(EmployeeID,LastName,FirstName,Title,TitleOfCourtesy,BirthDate,HireDate,Address,City,Region,PostalCode,Country,HomePhone,Extension,Photo,Notes,ReportsTo) VALUES(1,'Davolio','Nancy','Sales Representative','Ms.',stringdate('1948.12.08 00:00.00 000000'),stringdate('1992.05.01 00:00.00 000000'),'507 - 20th Ave. E.\15\12Apt. 2A','Seattle','WA','98122','United States','(206) 555-9857','5467',DB.DBA.get_blob_from_dav('/DAV/VAD/demo/sql/EMP1'), 'Education includes a BA in psychology from Colorado State University in 1970.  She also completed "The Art of the Cold Call."  Nancy is a member of Toastmasters International.',2);

INSERT soft Employees(EmployeeID,LastName,FirstName,Title,TitleOfCourtesy,BirthDate,HireDate,Address,City,Region,PostalCode,Country,HomePhone,Extension,Photo,Notes,ReportsTo) VALUES(2,'Fuller','Andrew','Vice President, Sales','Dr.',stringdate('1952.02.19 00:00.00 000000'),stringdate('1992.08.14 00:00.00 000000'),'908 W. Capital Way','Tacoma','WA','98401','United States','(206) 555-9482','3457',DB.DBA.get_blob_from_dav('/DAV/VAD/demo/sql/EMP2'), 'Andrew received his BTS commercial in 1974 and a Ph.D. in international marketing from the University of Dallas in 1981.  He is fluent in French and Italian and reads German.  He joined the company as a sales representative, was promoted to sales manager in January 1992 and to vice president of sales in March 1993.  Andrew is a member of the Sales Management Roundtable, the Seattle Chamber of Commerce, and the Pacific Rim Importers Association.',NULL);

INSERT soft Employees(EmployeeID,LastName,FirstName,Title,TitleOfCourtesy,BirthDate,HireDate,Address,City,Region,PostalCode,Country,HomePhone,Extension,Photo,Notes,ReportsTo) VALUES(3,'Leverling','Janet','Sales Representative','Ms.',stringdate('1963.08.30 00:00.00 000000'),stringdate('1992.04.01 00:00.00 000000'),'722 Moss Bay Blvd.','Kirkland','WA','98033','United States','(206) 555-3412','3355',DB.DBA.get_blob_from_dav('/DAV/VAD/demo/sql/EMP3'), 'Janet has a BS degree in chemistry from Boston College (1984).  She has also completed a certificate program in food retailing management.  Janet was hired as a sales associate in 1991 and promoted to sales representative in February 1992.',2);

INSERT soft Employees(EmployeeID,LastName,FirstName,Title,TitleOfCourtesy,BirthDate,HireDate,Address,City,Region,PostalCode,Country,HomePhone,Extension,Photo,Notes,ReportsTo) VALUES(4,'Peacock','Margaret','Sales Representative','Mrs.',stringdate('1937.09.19 00:00.00 000000'),stringdate('1993.05.03 00:00.00 000000'),'4110 Old Redmond Rd.','Redmond','WA','98052','United States','(206) 555-8122','5176',DB.DBA.get_blob_from_dav('/DAV/VAD/demo/sql/EMP4'), 'Margaret holds a BA in English literature from Concordia College (1958) and an MA from the American Institute of Culinary Arts (1966).  She was assigned to the London office temporarily from July through November 1992.',2);

INSERT soft Employees(EmployeeID,LastName,FirstName,Title,TitleOfCourtesy,BirthDate,HireDate,Address,City,Region,PostalCode,Country,HomePhone,Extension,Photo,Notes,ReportsTo) VALUES(5,'Buchanan','Steven','Sales Manager','Mr.',stringdate('1955.03.04 00:00.00 000000'),stringdate('1993.10.17 00:00.00 000000'),'14 Garrett Hill','London',NULL,'SW1 8JR','United Kingdom','(71) 555-4848','3453',DB.DBA.get_blob_from_dav('/DAV/VAD/demo/sql/EMP5'), 'Steven Buchanan graduated from St. Andrews University, Scotland, with a BSC degree in 1976.  Upon joining the company as a sales representative in 1992, he spent 6 months in an orientation program at the Seattle office and then returned to his permanent post in London.  He was promoted to sales manager in March 1993.  Mr. Buchanan has completed the courses "Successful Telemarketing" and "International Sales Management."  He is fluent in French.',2);

INSERT soft Employees(EmployeeID,LastName,FirstName,Title,TitleOfCourtesy,BirthDate,HireDate,Address,City,Region,PostalCode,Country,HomePhone,Extension,Photo,Notes,ReportsTo) VALUES(6,'Suyama','Michael','Sales Representative','Mr.',stringdate('1963.07.02 00:00.00 000000'),stringdate('1993.10.17 00:00.00 000000'),'Coventry House\15\12Miner Rd.','London',NULL,'EC2 7JR','United Kingdom','(71) 555-7773','428',DB.DBA.get_blob_from_dav('/DAV/VAD/demo/sql/EMP6'), 'Michael is a graduate of Sussex University (MA, economics, 1983) and the University of California at Los Angeles (MBA, marketing, 1986).  He has also taken the courses "Multi-Cultural Selling" and "Time Management for the Sales Professional."  He is fluent in Japanese and can read and write French, Portuguese, and Spanish.',5);

INSERT soft Employees(EmployeeID,LastName,FirstName,Title,TitleOfCourtesy,BirthDate,HireDate,Address,City,Region,PostalCode,Country,HomePhone,Extension,Photo,Notes,ReportsTo) VALUES(7,'King','Robert','Sales Representative','Mr.',stringdate('1960.05.29 00:00.00 000000'),stringdate('1994.01.02 00:00.00 000000'),'Edgeham Hollow\15\12Winchester Way','London',NULL,'RG1 9SP','United Kingdom','(71) 555-5598','465',DB.DBA.get_blob_from_dav('/DAV/VAD/demo/sql/EMP7'), 'Robert King served in the Peace Corps and traveled extensively before completing his degree in English at the University of Michigan in 1992, the year he joined the company.  After completing a course entitled "Selling in Europe," he was transferred to the London office in March 1993.',5);

INSERT soft Employees(EmployeeID,LastName,FirstName,Title,TitleOfCourtesy,BirthDate,HireDate,Address,City,Region,PostalCode,Country,HomePhone,Extension,Photo,Notes,ReportsTo) VALUES(8,'Callahan','Laura','Inside Sales Coordinator','Ms.',stringdate('1958.01.09 00:00.00 000000'),stringdate('1994.03.05 00:00.00 000000'),'4726 - 11th Ave. N.E.','Seattle','WA','98105','United States','(206) 555-1189','2344',DB.DBA.get_blob_from_dav('/DAV/VAD/demo/sql/EMP8'), 'Laura received a BA in psychology from the University of Washington.  She has also completed a course in business French.  She reads and writes French.',2);

INSERT soft Employees(EmployeeID,LastName,FirstName,Title,TitleOfCourtesy,BirthDate,HireDate,Address,City,Region,PostalCode,Country,HomePhone,Extension,Photo,Notes,ReportsTo) VALUES(9,'Dodsworth','Anne','Sales Representative','Ms.',stringdate('1966.01.27 00:00.00 000000'),stringdate('1994.11.15 00:00.00 000000'),'7 Houndstooth Rd.','London',NULL,'WG2 7LT','United Kingdom','(71) 555-4444','452',DB.DBA.get_blob_from_dav('/DAV/VAD/demo/sql/EMP9'), 'Anne has a BA degree in English from St. Lawrence College.  She is fluent in French and German.',5);

select count(*) from Employees;
--ECHO BOTH $IF $EQU $LAST[1] 9 "PASSED" "***FAILED";
--SET ARGV[$LIF] $+ $ARGV[$LIF] 1;
--ECHO BOTH ": " $LAST[1] " Employees loaded\n";

--
--
--

DB.DBA.exec_no_error('DELETE FROM Orders');

INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10330,'LILAS',3,stringdate('1994.11.16 00:00.00 000000'),stringdate('1994.12.14 00:00.00 000000'),stringdate('1994.11.28 00:00.00 000000'),1,12.750000,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10331,'BONAP',9,stringdate('1994.11.16 00:00.00 000000'),stringdate('1994.12.28 00:00.00 000000'),stringdate('1994.11.21 00:00.00 000000'),1,10.190000,'Bon app\47','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10332,'MEREP',3,stringdate('1994.11.17 00:00.00 000000'),stringdate('1994.12.29 00:00.00 000000'),stringdate('1994.11.21 00:00.00 000000'),2,52.840000,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10333,'WARTH',5,stringdate('1994.11.18 00:00.00 000000'),stringdate('1994.12.16 00:00.00 000000'),stringdate('1994.11.25 00:00.00 000000'),3,0.590000,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10334,'VICTE',8,stringdate('1994.11.21 00:00.00 000000'),stringdate('1994.12.19 00:00.00 000000'),stringdate('1994.11.28 00:00.00 000000'),2,8.560000,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10335,'HUNGO',7,stringdate('1994.11.22 00:00.00 000000'),stringdate('1994.12.20 00:00.00 000000'),stringdate('1994.11.24 00:00.00 000000'),2,42.110000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10336,'PRINI',7,stringdate('1994.11.23 00:00.00 000000'),stringdate('1994.12.21 00:00.00 000000'),stringdate('1994.11.25 00:00.00 000000'),2,15.510000,'Princesa Isabel Vinhos','Estrada da sade n. 58','Lisboa',NULL,'1756','Portugal');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10337,'FRANK',4,stringdate('1994.11.24 00:00.00 000000'),stringdate('1994.12.22 00:00.00 000000'),stringdate('1994.11.29 00:00.00 000000'),3,108.260000,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10338,'OLDWO',4,stringdate('1994.11.25 00:00.00 000000'),stringdate('1994.12.23 00:00.00 000000'),stringdate('1994.11.29 00:00.00 000000'),3,84.210000,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10339,'MEREP',2,stringdate('1994.11.28 00:00.00 000000'),stringdate('1994.12.26 00:00.00 000000'),stringdate('1994.12.05 00:00.00 000000'),2,15.660000,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10340,'BONAP',1,stringdate('1994.11.29 00:00.00 000000'),stringdate('1994.12.27 00:00.00 000000'),stringdate('1994.12.09 00:00.00 000000'),3,166.310000,'Bon app\47','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10341,'SIMOB',7,stringdate('1994.11.29 00:00.00 000000'),stringdate('1994.12.27 00:00.00 000000'),stringdate('1994.12.06 00:00.00 000000'),3,26.780000,'Simons bistro','Vinbltet 34','Kbenhavn',NULL,'1734','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10342,'FRANK',4,stringdate('1994.11.30 00:00.00 000000'),stringdate('1994.12.14 00:00.00 000000'),stringdate('1994.12.05 00:00.00 000000'),2,54.830000,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10343,'LEHMS',4,stringdate('1994.12.01 00:00.00 000000'),stringdate('1994.12.29 00:00.00 000000'),stringdate('1994.12.07 00:00.00 000000'),1,110.370000,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M. ',NULL,'60528','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10344,'WHITC',4,stringdate('1994.12.02 00:00.00 000000'),stringdate('1994.12.30 00:00.00 000000'),stringdate('1994.12.06 00:00.00 000000'),2,23.290000,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10345,'QUICK',2,stringdate('1994.12.05 00:00.00 000000'),stringdate('1995.01.02 00:00.00 000000'),stringdate('1994.12.12 00:00.00 000000'),2,249.060000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10346,'RATTC',3,stringdate('1994.12.06 00:00.00 000000'),stringdate('1995.01.17 00:00.00 000000'),stringdate('1994.12.09 00:00.00 000000'),3,142.080000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10248,'VINET',5,stringdate('1994.08.04 00:00.00 000000'),stringdate('1994.09.01 00:00.00 000000'),stringdate('1994.08.16 00:00.00 000000'),3,32.380000,'Vins et alcools Chevalier','59 rue de l\47Abbaye','Reims',NULL,'51100','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10249,'TOMSP',6,stringdate('1994.08.05 00:00.00 000000'),stringdate('1994.09.16 00:00.00 000000'),stringdate('1994.08.10 00:00.00 000000'),1,11.610000,'Toms Spezialitten','Luisenstr. 48','Mnster',NULL,'44087','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10250,'HANAR',4,stringdate('1994.08.08 00:00.00 000000'),stringdate('1994.09.05 00:00.00 000000'),stringdate('1994.08.12 00:00.00 000000'),2,65.830000,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10251,'VICTE',3,stringdate('1994.08.08 00:00.00 000000'),stringdate('1994.09.05 00:00.00 000000'),stringdate('1994.08.15 00:00.00 000000'),1,41.340000,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10252,'SUPRD',4,stringdate('1994.08.09 00:00.00 000000'),stringdate('1994.09.06 00:00.00 000000'),stringdate('1994.08.11 00:00.00 000000'),2,51.300000,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10253,'HANAR',3,stringdate('1994.08.10 00:00.00 000000'),stringdate('1994.08.24 00:00.00 000000'),stringdate('1994.08.16 00:00.00 000000'),2,58.170000,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10254,'CHOPS',5,stringdate('1994.08.11 00:00.00 000000'),stringdate('1994.09.08 00:00.00 000000'),stringdate('1994.08.23 00:00.00 000000'),2,22.980000,'Chop-suey Chinese','Hauptstr. 31','Bern',NULL,'3012','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10255,'RICSU',9,stringdate('1994.08.12 00:00.00 000000'),stringdate('1994.09.09 00:00.00 000000'),stringdate('1994.08.15 00:00.00 000000'),3,148.330000,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10256,'WELLI',3,stringdate('1994.08.15 00:00.00 000000'),stringdate('1994.09.12 00:00.00 000000'),stringdate('1994.08.17 00:00.00 000000'),2,13.970000,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10257,'HILAA',4,stringdate('1994.08.16 00:00.00 000000'),stringdate('1994.09.13 00:00.00 000000'),stringdate('1994.08.22 00:00.00 000000'),3,81.910000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10258,'ERNSH',1,stringdate('1994.08.17 00:00.00 000000'),stringdate('1994.09.14 00:00.00 000000'),stringdate('1994.08.23 00:00.00 000000'),1,140.510000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10259,'CENTC',4,stringdate('1994.08.18 00:00.00 000000'),stringdate('1994.09.15 00:00.00 000000'),stringdate('1994.08.25 00:00.00 000000'),3,3.250000,'Centro comercial Moctezuma','Sierras de Granada 9993','Mxico D.F.',NULL,'05022','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10260,'OTTIK',4,stringdate('1994.08.19 00:00.00 000000'),stringdate('1994.09.16 00:00.00 000000'),stringdate('1994.08.29 00:00.00 000000'),1,55.090000,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10261,'QUEDE',4,stringdate('1994.08.19 00:00.00 000000'),stringdate('1994.09.16 00:00.00 000000'),stringdate('1994.08.30 00:00.00 000000'),2,3.050000,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10262,'RATTC',8,stringdate('1994.08.22 00:00.00 000000'),stringdate('1994.09.19 00:00.00 000000'),stringdate('1994.08.25 00:00.00 000000'),3,48.290000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10263,'ERNSH',9,stringdate('1994.08.23 00:00.00 000000'),stringdate('1994.09.20 00:00.00 000000'),stringdate('1994.08.31 00:00.00 000000'),3,146.060000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10264,'FOLKO',6,stringdate('1994.08.24 00:00.00 000000'),stringdate('1994.09.21 00:00.00 000000'),stringdate('1994.09.23 00:00.00 000000'),3,3.670000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10265,'BLONP',2,stringdate('1994.08.25 00:00.00 000000'),stringdate('1994.09.22 00:00.00 000000'),stringdate('1994.09.12 00:00.00 000000'),1,55.280000,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10266,'WARTH',3,stringdate('1994.08.26 00:00.00 000000'),stringdate('1994.10.07 00:00.00 000000'),stringdate('1994.08.31 00:00.00 000000'),3,25.730000,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10267,'FRANK',4,stringdate('1994.08.29 00:00.00 000000'),stringdate('1994.09.26 00:00.00 000000'),stringdate('1994.09.06 00:00.00 000000'),1,208.580000,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10268,'GROSR',8,stringdate('1994.08.30 00:00.00 000000'),stringdate('1994.09.27 00:00.00 000000'),stringdate('1994.09.02 00:00.00 000000'),3,66.290000,'GROSELLA-Restaurante','5 Ave. Los Palos Grandes','Caracas','DF','1081','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10269,'WHITC',5,stringdate('1994.08.31 00:00.00 000000'),stringdate('1994.09.14 00:00.00 000000'),stringdate('1994.09.09 00:00.00 000000'),1,4.560000,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10270,'WARTH',1,stringdate('1994.09.01 00:00.00 000000'),stringdate('1994.09.29 00:00.00 000000'),stringdate('1994.09.02 00:00.00 000000'),1,136.540000,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10271,'SPLIR',6,stringdate('1994.09.01 00:00.00 000000'),stringdate('1994.09.29 00:00.00 000000'),stringdate('1994.09.30 00:00.00 000000'),2,4.540000,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10272,'RATTC',6,stringdate('1994.09.02 00:00.00 000000'),stringdate('1994.09.30 00:00.00 000000'),stringdate('1994.09.06 00:00.00 000000'),2,98.030000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10273,'QUICK',3,stringdate('1994.09.05 00:00.00 000000'),stringdate('1994.10.03 00:00.00 000000'),stringdate('1994.09.12 00:00.00 000000'),3,76.070000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10274,'VINET',6,stringdate('1994.09.06 00:00.00 000000'),stringdate('1994.10.04 00:00.00 000000'),stringdate('1994.09.16 00:00.00 000000'),1,6.010000,'Vins et alcools Chevalier','59 rue de l\47Abbaye','Reims',NULL,'51100','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10275,'MAGAA',1,stringdate('1994.09.07 00:00.00 000000'),stringdate('1994.10.05 00:00.00 000000'),stringdate('1994.09.09 00:00.00 000000'),1,26.930000,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10276,'TORTU',8,stringdate('1994.09.08 00:00.00 000000'),stringdate('1994.09.22 00:00.00 000000'),stringdate('1994.09.14 00:00.00 000000'),3,13.840000,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'05033','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10277,'MORGK',2,stringdate('1994.09.09 00:00.00 000000'),stringdate('1994.10.07 00:00.00 000000'),stringdate('1994.09.13 00:00.00 000000'),3,125.770000,'Morgenstern Gesundkost','Heerstr. 22','Leipzig',NULL,'04179','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10278,'BERGS',8,stringdate('1994.09.12 00:00.00 000000'),stringdate('1994.10.10 00:00.00 000000'),stringdate('1994.09.16 00:00.00 000000'),2,92.690000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10279,'LEHMS',8,stringdate('1994.09.13 00:00.00 000000'),stringdate('1994.10.11 00:00.00 000000'),stringdate('1994.09.16 00:00.00 000000'),2,25.830000,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M. ',NULL,'60528','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10280,'BERGS',2,stringdate('1994.09.14 00:00.00 000000'),stringdate('1994.10.12 00:00.00 000000'),stringdate('1994.10.13 00:00.00 000000'),1,8.980000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10281,'ROMEY',4,stringdate('1994.09.14 00:00.00 000000'),stringdate('1994.09.28 00:00.00 000000'),stringdate('1994.09.21 00:00.00 000000'),1,2.940000,'Romero y tomillo','Gran Va, 1','Madrid',NULL,'28001','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10282,'ROMEY',4,stringdate('1994.09.15 00:00.00 000000'),stringdate('1994.10.13 00:00.00 000000'),stringdate('1994.09.21 00:00.00 000000'),1,12.690000,'Romero y tomillo','Gran Va, 1','Madrid',NULL,'28001','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10283,'LILAS',3,stringdate('1994.09.16 00:00.00 000000'),stringdate('1994.10.14 00:00.00 000000'),stringdate('1994.09.23 00:00.00 000000'),3,84.810000,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10284,'LEHMS',4,stringdate('1994.09.19 00:00.00 000000'),stringdate('1994.10.17 00:00.00 000000'),stringdate('1994.09.27 00:00.00 000000'),1,76.560000,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M. ',NULL,'60528','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10285,'QUICK',1,stringdate('1994.09.20 00:00.00 000000'),stringdate('1994.10.18 00:00.00 000000'),stringdate('1994.09.26 00:00.00 000000'),2,76.830000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10286,'QUICK',8,stringdate('1994.09.21 00:00.00 000000'),stringdate('1994.10.19 00:00.00 000000'),stringdate('1994.09.30 00:00.00 000000'),3,229.240000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10287,'RICAR',8,stringdate('1994.09.22 00:00.00 000000'),stringdate('1994.10.20 00:00.00 000000'),stringdate('1994.09.28 00:00.00 000000'),3,12.760000,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10288,'REGGC',4,stringdate('1994.09.23 00:00.00 000000'),stringdate('1994.10.21 00:00.00 000000'),stringdate('1994.10.04 00:00.00 000000'),1,7.450000,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10289,'BSBEV',7,stringdate('1994.09.26 00:00.00 000000'),stringdate('1994.10.24 00:00.00 000000'),stringdate('1994.09.28 00:00.00 000000'),3,22.770000,'B\47s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10290,'COMMI',8,stringdate('1994.09.27 00:00.00 000000'),stringdate('1994.10.25 00:00.00 000000'),stringdate('1994.10.04 00:00.00 000000'),1,79.700000,'Comrcio Mineiro','Av. dos Lusadas, 23','So Paulo','SP','05432-043','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10291,'QUEDE',6,stringdate('1994.09.27 00:00.00 000000'),stringdate('1994.10.25 00:00.00 000000'),stringdate('1994.10.05 00:00.00 000000'),2,6.400000,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10292,'TRADH',1,stringdate('1994.09.28 00:00.00 000000'),stringdate('1994.10.26 00:00.00 000000'),stringdate('1994.10.03 00:00.00 000000'),2,1.350000,'Tradio Hipermercados','Av. Ins de Castro, 414','So Paulo','SP','05634-030','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10293,'TORTU',1,stringdate('1994.09.29 00:00.00 000000'),stringdate('1994.10.27 00:00.00 000000'),stringdate('1994.10.12 00:00.00 000000'),3,21.180000,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'05033','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10294,'RATTC',4,stringdate('1994.09.30 00:00.00 000000'),stringdate('1994.10.28 00:00.00 000000'),stringdate('1994.10.06 00:00.00 000000'),2,147.260000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10295,'VINET',2,stringdate('1994.10.03 00:00.00 000000'),stringdate('1994.10.31 00:00.00 000000'),stringdate('1994.10.11 00:00.00 000000'),2,1.150000,'Vins et alcools Chevalier','59 rue de l\47Abbaye','Reims',NULL,'51100','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10296,'LILAS',6,stringdate('1994.10.04 00:00.00 000000'),stringdate('1994.11.01 00:00.00 000000'),stringdate('1994.10.12 00:00.00 000000'),1,0.120000,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10297,'BLONP',5,stringdate('1994.10.05 00:00.00 000000'),stringdate('1994.11.16 00:00.00 000000'),stringdate('1994.10.11 00:00.00 000000'),2,5.740000,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10298,'HUNGO',6,stringdate('1994.10.06 00:00.00 000000'),stringdate('1994.11.03 00:00.00 000000'),stringdate('1994.10.12 00:00.00 000000'),2,168.220000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10299,'RICAR',4,stringdate('1994.10.07 00:00.00 000000'),stringdate('1994.11.04 00:00.00 000000'),stringdate('1994.10.14 00:00.00 000000'),2,29.760000,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10300,'MAGAA',2,stringdate('1994.10.10 00:00.00 000000'),stringdate('1994.11.07 00:00.00 000000'),stringdate('1994.10.19 00:00.00 000000'),2,17.680000,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10301,'WANDK',8,stringdate('1994.10.10 00:00.00 000000'),stringdate('1994.11.07 00:00.00 000000'),stringdate('1994.10.18 00:00.00 000000'),2,45.080000,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10302,'SUPRD',4,stringdate('1994.10.11 00:00.00 000000'),stringdate('1994.11.08 00:00.00 000000'),stringdate('1994.11.09 00:00.00 000000'),2,6.270000,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10303,'GODOS',7,stringdate('1994.10.12 00:00.00 000000'),stringdate('1994.11.09 00:00.00 000000'),stringdate('1994.10.19 00:00.00 000000'),2,107.830000,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10304,'TORTU',1,stringdate('1994.10.13 00:00.00 000000'),stringdate('1994.11.10 00:00.00 000000'),stringdate('1994.10.18 00:00.00 000000'),2,63.790000,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'05033','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10305,'OLDWO',8,stringdate('1994.10.14 00:00.00 000000'),stringdate('1994.11.11 00:00.00 000000'),stringdate('1994.11.09 00:00.00 000000'),3,257.620000,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10306,'ROMEY',1,stringdate('1994.10.17 00:00.00 000000'),stringdate('1994.11.14 00:00.00 000000'),stringdate('1994.10.24 00:00.00 000000'),3,7.560000,'Romero y tomillo','Gran Va, 1','Madrid',NULL,'28001','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10307,'LONEP',2,stringdate('1994.10.18 00:00.00 000000'),stringdate('1994.11.15 00:00.00 000000'),stringdate('1994.10.26 00:00.00 000000'),2,0.560000,'Lonesome Pine Restaurant','89 Chiaroscuro Rd.','Portland','OR','97219','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10308,'ANATR',7,stringdate('1994.10.19 00:00.00 000000'),stringdate('1994.11.16 00:00.00 000000'),stringdate('1994.10.25 00:00.00 000000'),3,1.610000,'Ana Trujillo Emparedados y helados','Avda. de la Constitucin 2222','Mxico D.F.',NULL,'05021','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10309,'HUNGO',3,stringdate('1994.10.20 00:00.00 000000'),stringdate('1994.11.17 00:00.00 000000'),stringdate('1994.11.23 00:00.00 000000'),1,47.300000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10310,'THEBI',8,stringdate('1994.10.21 00:00.00 000000'),stringdate('1994.11.18 00:00.00 000000'),stringdate('1994.10.28 00:00.00 000000'),2,17.520000,'The Big Cheese','89 Jefferson Way\15\12Suite 2','Portland','OR','97201','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10311,'DUMON',1,stringdate('1994.10.21 00:00.00 000000'),stringdate('1994.11.04 00:00.00 000000'),stringdate('1994.10.27 00:00.00 000000'),3,24.690000,'Du monde entier','67, rue des Cinquante Otages','Nantes',NULL,'44000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10312,'WANDK',2,stringdate('1994.10.24 00:00.00 000000'),stringdate('1994.11.21 00:00.00 000000'),stringdate('1994.11.03 00:00.00 000000'),2,40.260000,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10313,'QUICK',2,stringdate('1994.10.25 00:00.00 000000'),stringdate('1994.11.22 00:00.00 000000'),stringdate('1994.11.04 00:00.00 000000'),2,1.960000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10314,'RATTC',1,stringdate('1994.10.26 00:00.00 000000'),stringdate('1994.11.23 00:00.00 000000'),stringdate('1994.11.04 00:00.00 000000'),2,74.160000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10315,'ISLAT',4,stringdate('1994.10.27 00:00.00 000000'),stringdate('1994.11.24 00:00.00 000000'),stringdate('1994.11.03 00:00.00 000000'),2,41.760000,'Island Trading','Garden House\15\12Crowther Way','Cowes','Isle of Wight','PO31 7PJ','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10316,'RATTC',1,stringdate('1994.10.28 00:00.00 000000'),stringdate('1994.11.25 00:00.00 000000'),stringdate('1994.11.08 00:00.00 000000'),3,150.150000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10317,'LONEP',6,stringdate('1994.10.31 00:00.00 000000'),stringdate('1994.11.28 00:00.00 000000'),stringdate('1994.11.10 00:00.00 000000'),1,12.690000,'Lonesome Pine Restaurant','89 Chiaroscuro Rd.','Portland','OR','97219','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10318,'ISLAT',8,stringdate('1994.11.01 00:00.00 000000'),stringdate('1994.11.29 00:00.00 000000'),stringdate('1994.11.04 00:00.00 000000'),2,4.730000,'Island Trading','Garden House\15\12Crowther Way','Cowes','Isle of Wight','PO31 7PJ','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10319,'TORTU',7,stringdate('1994.11.02 00:00.00 000000'),stringdate('1994.11.30 00:00.00 000000'),stringdate('1994.11.11 00:00.00 000000'),3,64.500000,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'05033','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10320,'WARTH',5,stringdate('1994.11.03 00:00.00 000000'),stringdate('1994.11.17 00:00.00 000000'),stringdate('1994.11.18 00:00.00 000000'),3,34.570000,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10321,'ISLAT',3,stringdate('1994.11.03 00:00.00 000000'),stringdate('1994.12.01 00:00.00 000000'),stringdate('1994.11.11 00:00.00 000000'),2,3.430000,'Island Trading','Garden House\15\12Crowther Way','Cowes','Isle of Wight','PO31 7PJ','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10322,'PERIC',7,stringdate('1994.11.04 00:00.00 000000'),stringdate('1994.12.02 00:00.00 000000'),stringdate('1994.11.23 00:00.00 000000'),3,0.400000,'Pericles Comidas clsicas','Calle Dr. Jorge Cash 321','Mxico D.F.',NULL,'05033','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10323,'KOENE',4,stringdate('1994.11.07 00:00.00 000000'),stringdate('1994.12.05 00:00.00 000000'),stringdate('1994.11.14 00:00.00 000000'),1,4.880000,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10324,'SAVEA',9,stringdate('1994.11.08 00:00.00 000000'),stringdate('1994.12.06 00:00.00 000000'),stringdate('1994.11.10 00:00.00 000000'),1,214.270000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10325,'KOENE',1,stringdate('1994.11.09 00:00.00 000000'),stringdate('1994.11.23 00:00.00 000000'),stringdate('1994.11.14 00:00.00 000000'),3,64.860000,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10326,'BOLID',4,stringdate('1994.11.10 00:00.00 000000'),stringdate('1994.12.08 00:00.00 000000'),stringdate('1994.11.14 00:00.00 000000'),2,77.920000,'Blido Comidas preparadas','C/ Araquil, 67','Madrid',NULL,'28023','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10327,'FOLKO',2,stringdate('1994.11.11 00:00.00 000000'),stringdate('1994.12.09 00:00.00 000000'),stringdate('1994.11.14 00:00.00 000000'),1,63.360000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10328,'FURIB',4,stringdate('1994.11.14 00:00.00 000000'),stringdate('1994.12.12 00:00.00 000000'),stringdate('1994.11.17 00:00.00 000000'),3,87.030000,'Furia Bacalhau e Frutos do Mar','Jardim das rosas n. 32','Lisboa',NULL,'1675','Portugal');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10329,'SPLIR',4,stringdate('1994.11.15 00:00.00 000000'),stringdate('1994.12.27 00:00.00 000000'),stringdate('1994.11.23 00:00.00 000000'),2,191.670000,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10347,'FAMIA',4,stringdate('1994.12.07 00:00.00 000000'),stringdate('1995.01.04 00:00.00 000000'),stringdate('1994.12.09 00:00.00 000000'),3,3.100000,'Familia Arquibaldo','Rua Ors, 92','So Paulo','SP','05442-030','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10348,'WANDK',4,stringdate('1994.12.08 00:00.00 000000'),stringdate('1995.01.05 00:00.00 000000'),stringdate('1994.12.16 00:00.00 000000'),2,0.780000,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10349,'SPLIR',7,stringdate('1994.12.09 00:00.00 000000'),stringdate('1995.01.06 00:00.00 000000'),stringdate('1994.12.16 00:00.00 000000'),1,8.630000,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10350,'LAMAI',6,stringdate('1994.12.12 00:00.00 000000'),stringdate('1995.01.09 00:00.00 000000'),stringdate('1995.01.03 00:00.00 000000'),2,64.190000,'La maison d\47Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10351,'ERNSH',1,stringdate('1994.12.12 00:00.00 000000'),stringdate('1995.01.09 00:00.00 000000'),stringdate('1994.12.21 00:00.00 000000'),1,162.330000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10352,'FURIB',3,stringdate('1994.12.13 00:00.00 000000'),stringdate('1994.12.27 00:00.00 000000'),stringdate('1994.12.19 00:00.00 000000'),3,1.300000,'Furia Bacalhau e Frutos do Mar','Jardim das rosas n. 32','Lisboa',NULL,'1675','Portugal');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10353,'PICCO',7,stringdate('1994.12.14 00:00.00 000000'),stringdate('1995.01.11 00:00.00 000000'),stringdate('1994.12.26 00:00.00 000000'),3,360.630000,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10354,'PERIC',8,stringdate('1994.12.15 00:00.00 000000'),stringdate('1995.01.12 00:00.00 000000'),stringdate('1994.12.21 00:00.00 000000'),3,53.800000,'Pericles Comidas clsicas','Calle Dr. Jorge Cash 321','Mxico D.F.',NULL,'05033','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10355,'AROUT',6,stringdate('1994.12.16 00:00.00 000000'),stringdate('1995.01.13 00:00.00 000000'),stringdate('1994.12.21 00:00.00 000000'),1,41.950000,'Around the Horn','Brook Farm\15\12Stratford St. Mary','Colchester','Essex','CO7 6JX','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10356,'WANDK',6,stringdate('1994.12.19 00:00.00 000000'),stringdate('1995.01.16 00:00.00 000000'),stringdate('1994.12.28 00:00.00 000000'),2,36.710000,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10357,'LILAS',1,stringdate('1994.12.20 00:00.00 000000'),stringdate('1995.01.17 00:00.00 000000'),stringdate('1995.01.02 00:00.00 000000'),3,34.880000,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10358,'LAMAI',5,stringdate('1994.12.21 00:00.00 000000'),stringdate('1995.01.18 00:00.00 000000'),stringdate('1994.12.28 00:00.00 000000'),1,19.640000,'La maison d\47Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10359,'SEVES',5,stringdate('1994.12.22 00:00.00 000000'),stringdate('1995.01.19 00:00.00 000000'),stringdate('1994.12.27 00:00.00 000000'),3,288.430000,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10360,'BLONP',4,stringdate('1994.12.23 00:00.00 000000'),stringdate('1995.01.20 00:00.00 000000'),stringdate('1995.01.02 00:00.00 000000'),3,131.700000,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10361,'QUICK',1,stringdate('1994.12.23 00:00.00 000000'),stringdate('1995.01.20 00:00.00 000000'),stringdate('1995.01.03 00:00.00 000000'),2,183.170000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10362,'BONAP',3,stringdate('1994.12.26 00:00.00 000000'),stringdate('1995.01.23 00:00.00 000000'),stringdate('1994.12.29 00:00.00 000000'),1,96.040000,'Bon app\47','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10363,'DRACD',4,stringdate('1994.12.27 00:00.00 000000'),stringdate('1995.01.24 00:00.00 000000'),stringdate('1995.01.04 00:00.00 000000'),3,30.540000,'Drachenblut Delikatessen','Walserweg 21','Aachen',NULL,'52066','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10364,'EASTC',1,stringdate('1994.12.27 00:00.00 000000'),stringdate('1995.02.07 00:00.00 000000'),stringdate('1995.01.04 00:00.00 000000'),1,71.970000,'Eastern Connection','35 King George','London',NULL,'WX3 6FW','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10365,'ANTON',3,stringdate('1994.12.28 00:00.00 000000'),stringdate('1995.01.25 00:00.00 000000'),stringdate('1995.01.02 00:00.00 000000'),2,22.000000,'Antonio Moreno Taquera','Mataderos  2312','Mxico D.F.',NULL,'05023','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10366,'GALED',8,stringdate('1994.12.29 00:00.00 000000'),stringdate('1995.02.09 00:00.00 000000'),stringdate('1995.01.30 00:00.00 000000'),2,10.140000,'Galera del gastronmo','Rambla de Catalua, 23','Barcelona',NULL,'8022','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10367,'VAFFE',7,stringdate('1994.12.29 00:00.00 000000'),stringdate('1995.01.26 00:00.00 000000'),stringdate('1995.01.02 00:00.00 000000'),3,13.550000,'Vaffeljernet','Smagslget 45','rhus',NULL,'8200','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10368,'ERNSH',2,stringdate('1994.12.30 00:00.00 000000'),stringdate('1995.01.27 00:00.00 000000'),stringdate('1995.01.02 00:00.00 000000'),2,101.950000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10369,'SPLIR',8,stringdate('1995.01.02 00:00.00 000000'),stringdate('1995.01.30 00:00.00 000000'),stringdate('1995.01.09 00:00.00 000000'),2,195.680000,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10370,'CHOPS',6,stringdate('1995.01.03 00:00.00 000000'),stringdate('1995.01.31 00:00.00 000000'),stringdate('1995.01.27 00:00.00 000000'),2,1.170000,'Chop-suey Chinese','Hauptstr. 31','Bern',NULL,'3012','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10371,'LAMAI',1,stringdate('1995.01.03 00:00.00 000000'),stringdate('1995.01.31 00:00.00 000000'),stringdate('1995.01.24 00:00.00 000000'),1,0.450000,'La maison d\47Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10372,'QUEEN',5,stringdate('1995.01.04 00:00.00 000000'),stringdate('1995.02.01 00:00.00 000000'),stringdate('1995.01.09 00:00.00 000000'),2,890.780000,'Queen Cozinha','Alameda dos Canrios, 891','So Paulo','SP','05487-020','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10373,'HUNGO',4,stringdate('1995.01.05 00:00.00 000000'),stringdate('1995.02.02 00:00.00 000000'),stringdate('1995.01.11 00:00.00 000000'),3,124.120000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10374,'WOLZA',1,stringdate('1995.01.05 00:00.00 000000'),stringdate('1995.02.02 00:00.00 000000'),stringdate('1995.01.09 00:00.00 000000'),3,3.940000,'Wolski Zajazd','ul. Filtrowa 68','Warszawa',NULL,'01-012','Poland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10375,'HUNGC',3,stringdate('1995.01.06 00:00.00 000000'),stringdate('1995.02.03 00:00.00 000000'),stringdate('1995.01.09 00:00.00 000000'),2,20.120000,'Hungry Coyote Import Store','City Center Plaza\15\12516 Main St.','Elgin','OR','97827','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10376,'MEREP',1,stringdate('1995.01.09 00:00.00 000000'),stringdate('1995.02.06 00:00.00 000000'),stringdate('1995.01.13 00:00.00 000000'),2,20.390000,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10377,'SEVES',1,stringdate('1995.01.09 00:00.00 000000'),stringdate('1995.02.06 00:00.00 000000'),stringdate('1995.01.13 00:00.00 000000'),3,22.210000,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10378,'FOLKO',5,stringdate('1995.01.10 00:00.00 000000'),stringdate('1995.02.07 00:00.00 000000'),stringdate('1995.01.19 00:00.00 000000'),3,5.440000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10379,'QUEDE',2,stringdate('1995.01.11 00:00.00 000000'),stringdate('1995.02.08 00:00.00 000000'),stringdate('1995.01.13 00:00.00 000000'),1,45.030000,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10380,'HUNGO',8,stringdate('1995.01.12 00:00.00 000000'),stringdate('1995.02.09 00:00.00 000000'),stringdate('1995.02.16 00:00.00 000000'),3,35.030000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10381,'LILAS',3,stringdate('1995.01.12 00:00.00 000000'),stringdate('1995.02.09 00:00.00 000000'),stringdate('1995.01.13 00:00.00 000000'),3,7.990000,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10382,'ERNSH',4,stringdate('1995.01.13 00:00.00 000000'),stringdate('1995.02.10 00:00.00 000000'),stringdate('1995.01.16 00:00.00 000000'),1,94.770000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10383,'AROUT',8,stringdate('1995.01.16 00:00.00 000000'),stringdate('1995.02.13 00:00.00 000000'),stringdate('1995.01.18 00:00.00 000000'),3,34.240000,'Around the Horn','Brook Farm\15\12Stratford St. Mary','Colchester','Essex','CO7 6JX','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10384,'BERGS',3,stringdate('1995.01.16 00:00.00 000000'),stringdate('1995.02.13 00:00.00 000000'),stringdate('1995.01.20 00:00.00 000000'),3,168.640000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10385,'SPLIR',1,stringdate('1995.01.17 00:00.00 000000'),stringdate('1995.02.14 00:00.00 000000'),stringdate('1995.01.23 00:00.00 000000'),2,30.960000,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10386,'FAMIA',9,stringdate('1995.01.18 00:00.00 000000'),stringdate('1995.02.01 00:00.00 000000'),stringdate('1995.01.25 00:00.00 000000'),3,13.990000,'Familia Arquibaldo','Rua Ors, 92','So Paulo','SP','05442-030','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10387,'SANTG',1,stringdate('1995.01.18 00:00.00 000000'),stringdate('1995.02.15 00:00.00 000000'),stringdate('1995.01.20 00:00.00 000000'),2,93.630000,'Sant Gourmet','Erling Skakkes gate 78','Stavern',NULL,'4110','Norway');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10388,'SEVES',2,stringdate('1995.01.19 00:00.00 000000'),stringdate('1995.02.16 00:00.00 000000'),stringdate('1995.01.20 00:00.00 000000'),1,34.860000,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10389,'BOTTM',4,stringdate('1995.01.20 00:00.00 000000'),stringdate('1995.02.17 00:00.00 000000'),stringdate('1995.01.24 00:00.00 000000'),2,47.420000,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10390,'ERNSH',6,stringdate('1995.01.23 00:00.00 000000'),stringdate('1995.02.20 00:00.00 000000'),stringdate('1995.01.26 00:00.00 000000'),1,126.380000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10391,'DRACD',3,stringdate('1995.01.23 00:00.00 000000'),stringdate('1995.02.20 00:00.00 000000'),stringdate('1995.01.31 00:00.00 000000'),3,5.450000,'Drachenblut Delikatessen','Walserweg 21','Aachen',NULL,'52066','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10392,'PICCO',2,stringdate('1995.01.24 00:00.00 000000'),stringdate('1995.02.21 00:00.00 000000'),stringdate('1995.02.01 00:00.00 000000'),3,122.460000,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10393,'SAVEA',1,stringdate('1995.01.25 00:00.00 000000'),stringdate('1995.02.22 00:00.00 000000'),stringdate('1995.02.03 00:00.00 000000'),3,126.560000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10394,'HUNGC',1,stringdate('1995.01.25 00:00.00 000000'),stringdate('1995.02.22 00:00.00 000000'),stringdate('1995.02.03 00:00.00 000000'),3,30.340000,'Hungry Coyote Import Store','City Center Plaza\15\12516 Main St.','Elgin','OR','97827','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10395,'HILAA',6,stringdate('1995.01.26 00:00.00 000000'),stringdate('1995.02.23 00:00.00 000000'),stringdate('1995.02.03 00:00.00 000000'),1,184.410000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10396,'FRANK',1,stringdate('1995.01.27 00:00.00 000000'),stringdate('1995.02.10 00:00.00 000000'),stringdate('1995.02.06 00:00.00 000000'),3,135.350000,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10397,'PRINI',5,stringdate('1995.01.27 00:00.00 000000'),stringdate('1995.02.24 00:00.00 000000'),stringdate('1995.02.02 00:00.00 000000'),1,60.260000,'Princesa Isabel Vinhos','Estrada da sade n. 58','Lisboa',NULL,'1756','Portugal');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10398,'SAVEA',2,stringdate('1995.01.30 00:00.00 000000'),stringdate('1995.02.27 00:00.00 000000'),stringdate('1995.02.09 00:00.00 000000'),3,89.160000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10399,'VAFFE',8,stringdate('1995.01.31 00:00.00 000000'),stringdate('1995.02.14 00:00.00 000000'),stringdate('1995.02.08 00:00.00 000000'),3,27.360000,'Vaffeljernet','Smagslget 45','rhus',NULL,'8200','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10400,'EASTC',1,stringdate('1995.02.01 00:00.00 000000'),stringdate('1995.03.01 00:00.00 000000'),stringdate('1995.02.16 00:00.00 000000'),3,83.930000,'Eastern Connection','35 King George','London',NULL,'WX3 6FW','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10401,'RATTC',1,stringdate('1995.02.01 00:00.00 000000'),stringdate('1995.03.01 00:00.00 000000'),stringdate('1995.02.10 00:00.00 000000'),1,12.510000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10402,'ERNSH',8,stringdate('1995.02.02 00:00.00 000000'),stringdate('1995.03.16 00:00.00 000000'),stringdate('1995.02.10 00:00.00 000000'),2,67.880000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10403,'ERNSH',4,stringdate('1995.02.03 00:00.00 000000'),stringdate('1995.03.03 00:00.00 000000'),stringdate('1995.02.09 00:00.00 000000'),3,73.790000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10404,'MAGAA',2,stringdate('1995.02.03 00:00.00 000000'),stringdate('1995.03.03 00:00.00 000000'),stringdate('1995.02.08 00:00.00 000000'),1,155.970000,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10405,'LINOD',1,stringdate('1995.02.06 00:00.00 000000'),stringdate('1995.03.06 00:00.00 000000'),stringdate('1995.02.22 00:00.00 000000'),1,34.820000,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10406,'QUEEN',7,stringdate('1995.02.07 00:00.00 000000'),stringdate('1995.03.21 00:00.00 000000'),stringdate('1995.02.13 00:00.00 000000'),1,108.040000,'Queen Cozinha','Alameda dos Canrios, 891','So Paulo','SP','05487-020','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10407,'OTTIK',2,stringdate('1995.02.07 00:00.00 000000'),stringdate('1995.03.07 00:00.00 000000'),stringdate('1995.03.02 00:00.00 000000'),2,91.480000,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10408,'FOLIG',8,stringdate('1995.02.08 00:00.00 000000'),stringdate('1995.03.08 00:00.00 000000'),stringdate('1995.02.14 00:00.00 000000'),1,11.260000,'Folies gourmandes','184, chausse de Tournai','Lille',NULL,'59000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10409,'OCEAN',3,stringdate('1995.02.09 00:00.00 000000'),stringdate('1995.03.09 00:00.00 000000'),stringdate('1995.02.14 00:00.00 000000'),1,29.830000,'Ocano Atlntico Ltda.','Ing. Gustavo Moncada 8585\15\12Piso 20-A','Buenos Aires',NULL,'1010','Argentina');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10410,'BOTTM',3,stringdate('1995.02.10 00:00.00 000000'),stringdate('1995.03.10 00:00.00 000000'),stringdate('1995.02.15 00:00.00 000000'),3,2.400000,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10411,'BOTTM',9,stringdate('1995.02.10 00:00.00 000000'),stringdate('1995.03.10 00:00.00 000000'),stringdate('1995.02.21 00:00.00 000000'),3,23.650000,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10412,'WARTH',8,stringdate('1995.02.13 00:00.00 000000'),stringdate('1995.03.13 00:00.00 000000'),stringdate('1995.02.15 00:00.00 000000'),2,3.770000,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10413,'LAMAI',3,stringdate('1995.02.14 00:00.00 000000'),stringdate('1995.03.14 00:00.00 000000'),stringdate('1995.02.16 00:00.00 000000'),2,95.660000,'La maison d\47Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10414,'FAMIA',2,stringdate('1995.02.14 00:00.00 000000'),stringdate('1995.03.14 00:00.00 000000'),stringdate('1995.02.17 00:00.00 000000'),3,21.480000,'Familia Arquibaldo','Rua Ors, 92','So Paulo','SP','05442-030','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10415,'HUNGC',3,stringdate('1995.02.15 00:00.00 000000'),stringdate('1995.03.15 00:00.00 000000'),stringdate('1995.02.24 00:00.00 000000'),1,0.200000,'Hungry Coyote Import Store','City Center Plaza\15\12516 Main St.','Elgin','OR','97827','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10416,'WARTH',8,stringdate('1995.02.16 00:00.00 000000'),stringdate('1995.03.16 00:00.00 000000'),stringdate('1995.02.27 00:00.00 000000'),3,22.720000,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10417,'SIMOB',4,stringdate('1995.02.16 00:00.00 000000'),stringdate('1995.03.16 00:00.00 000000'),stringdate('1995.02.28 00:00.00 000000'),3,70.290000,'Simons bistro','Vinbltet 34','Kbenhavn',NULL,'1734','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10418,'QUICK',4,stringdate('1995.02.17 00:00.00 000000'),stringdate('1995.03.17 00:00.00 000000'),stringdate('1995.02.24 00:00.00 000000'),1,17.550000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10419,'RICSU',4,stringdate('1995.02.20 00:00.00 000000'),stringdate('1995.03.20 00:00.00 000000'),stringdate('1995.03.02 00:00.00 000000'),2,137.350000,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10420,'WELLI',3,stringdate('1995.02.21 00:00.00 000000'),stringdate('1995.03.21 00:00.00 000000'),stringdate('1995.02.27 00:00.00 000000'),1,44.120000,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10421,'QUEDE',8,stringdate('1995.02.21 00:00.00 000000'),stringdate('1995.04.04 00:00.00 000000'),stringdate('1995.02.27 00:00.00 000000'),1,99.230000,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10422,'FRANS',2,stringdate('1995.02.22 00:00.00 000000'),stringdate('1995.03.22 00:00.00 000000'),stringdate('1995.03.03 00:00.00 000000'),1,3.020000,'Franchi S.p.A.','Via Monte Bianco 34','Torino',NULL,'10100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10423,'GOURL',6,stringdate('1995.02.23 00:00.00 000000'),stringdate('1995.03.09 00:00.00 000000'),stringdate('1995.03.27 00:00.00 000000'),3,24.500000,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10424,'MEREP',7,stringdate('1995.02.23 00:00.00 000000'),stringdate('1995.03.23 00:00.00 000000'),stringdate('1995.02.27 00:00.00 000000'),2,370.610000,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10425,'LAMAI',6,stringdate('1995.02.24 00:00.00 000000'),stringdate('1995.03.24 00:00.00 000000'),stringdate('1995.03.17 00:00.00 000000'),2,7.930000,'La maison d\47Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10426,'GALED',4,stringdate('1995.02.27 00:00.00 000000'),stringdate('1995.03.27 00:00.00 000000'),stringdate('1995.03.09 00:00.00 000000'),1,18.690000,'Galera del gastronmo','Rambla de Catalua, 23','Barcelona',NULL,'8022','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10427,'PICCO',4,stringdate('1995.02.27 00:00.00 000000'),stringdate('1995.03.27 00:00.00 000000'),stringdate('1995.04.03 00:00.00 000000'),2,31.290000,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10428,'REGGC',7,stringdate('1995.02.28 00:00.00 000000'),stringdate('1995.03.28 00:00.00 000000'),stringdate('1995.03.07 00:00.00 000000'),1,11.090000,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10429,'HUNGO',3,stringdate('1995.03.01 00:00.00 000000'),stringdate('1995.04.12 00:00.00 000000'),stringdate('1995.03.10 00:00.00 000000'),2,56.630000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10430,'ERNSH',4,stringdate('1995.03.02 00:00.00 000000'),stringdate('1995.03.16 00:00.00 000000'),stringdate('1995.03.06 00:00.00 000000'),1,458.780000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10431,'BOTTM',4,stringdate('1995.03.02 00:00.00 000000'),stringdate('1995.03.16 00:00.00 000000'),stringdate('1995.03.10 00:00.00 000000'),2,44.170000,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10432,'SPLIR',3,stringdate('1995.03.03 00:00.00 000000'),stringdate('1995.03.17 00:00.00 000000'),stringdate('1995.03.10 00:00.00 000000'),2,4.340000,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10433,'PRINI',3,stringdate('1995.03.06 00:00.00 000000'),stringdate('1995.04.03 00:00.00 000000'),stringdate('1995.04.04 00:00.00 000000'),3,73.830000,'Princesa Isabel Vinhos','Estrada da sade n. 58','Lisboa',NULL,'1756','Portugal');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10434,'FOLKO',3,stringdate('1995.03.06 00:00.00 000000'),stringdate('1995.04.03 00:00.00 000000'),stringdate('1995.03.16 00:00.00 000000'),2,17.920000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10435,'CONSH',8,stringdate('1995.03.07 00:00.00 000000'),stringdate('1995.04.18 00:00.00 000000'),stringdate('1995.03.10 00:00.00 000000'),2,9.210000,'Consolidated Holdings','Berkeley Gardens\15\1212  Brewery ','London',NULL,'WX1 6LT','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10436,'BLONP',3,stringdate('1995.03.08 00:00.00 000000'),stringdate('1995.04.05 00:00.00 000000'),stringdate('1995.03.14 00:00.00 000000'),2,156.660000,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10437,'WARTH',8,stringdate('1995.03.08 00:00.00 000000'),stringdate('1995.04.05 00:00.00 000000'),stringdate('1995.03.15 00:00.00 000000'),1,19.970000,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10438,'TOMSP',3,stringdate('1995.03.09 00:00.00 000000'),stringdate('1995.04.06 00:00.00 000000'),stringdate('1995.03.17 00:00.00 000000'),2,8.240000,'Toms Spezialitten','Luisenstr. 48','Mnster',NULL,'44087','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10439,'MEREP',6,stringdate('1995.03.10 00:00.00 000000'),stringdate('1995.04.07 00:00.00 000000'),stringdate('1995.03.13 00:00.00 000000'),3,4.070000,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10440,'SAVEA',4,stringdate('1995.03.13 00:00.00 000000'),stringdate('1995.04.10 00:00.00 000000'),stringdate('1995.03.31 00:00.00 000000'),2,86.530000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10441,'OLDWO',3,stringdate('1995.03.13 00:00.00 000000'),stringdate('1995.04.24 00:00.00 000000'),stringdate('1995.04.14 00:00.00 000000'),2,73.020000,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10442,'ERNSH',3,stringdate('1995.03.14 00:00.00 000000'),stringdate('1995.04.11 00:00.00 000000'),stringdate('1995.03.21 00:00.00 000000'),2,47.940000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10443,'REGGC',8,stringdate('1995.03.15 00:00.00 000000'),stringdate('1995.04.12 00:00.00 000000'),stringdate('1995.03.17 00:00.00 000000'),1,13.950000,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10444,'BERGS',3,stringdate('1995.03.15 00:00.00 000000'),stringdate('1995.04.12 00:00.00 000000'),stringdate('1995.03.24 00:00.00 000000'),3,3.500000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10445,'BERGS',3,stringdate('1995.03.16 00:00.00 000000'),stringdate('1995.04.13 00:00.00 000000'),stringdate('1995.03.23 00:00.00 000000'),1,9.300000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10446,'TOMSP',6,stringdate('1995.03.17 00:00.00 000000'),stringdate('1995.04.14 00:00.00 000000'),stringdate('1995.03.22 00:00.00 000000'),1,14.680000,'Toms Spezialitten','Luisenstr. 48','Mnster',NULL,'44087','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10447,'RICAR',4,stringdate('1995.03.17 00:00.00 000000'),stringdate('1995.04.14 00:00.00 000000'),stringdate('1995.04.07 00:00.00 000000'),2,68.660000,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10448,'RANCH',4,stringdate('1995.03.20 00:00.00 000000'),stringdate('1995.04.17 00:00.00 000000'),stringdate('1995.03.27 00:00.00 000000'),2,38.820000,'Rancho grande','Av. del Libertador 900','Buenos Aires',NULL,'1010','Argentina');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10449,'BLONP',3,stringdate('1995.03.21 00:00.00 000000'),stringdate('1995.04.18 00:00.00 000000'),stringdate('1995.03.30 00:00.00 000000'),2,53.300000,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10450,'VICTE',8,stringdate('1995.03.22 00:00.00 000000'),stringdate('1995.04.19 00:00.00 000000'),stringdate('1995.04.11 00:00.00 000000'),2,7.230000,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10451,'QUICK',4,stringdate('1995.03.22 00:00.00 000000'),stringdate('1995.04.05 00:00.00 000000'),stringdate('1995.04.12 00:00.00 000000'),3,189.090000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10452,'SAVEA',8,stringdate('1995.03.23 00:00.00 000000'),stringdate('1995.04.20 00:00.00 000000'),stringdate('1995.03.29 00:00.00 000000'),1,140.260000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10453,'AROUT',1,stringdate('1995.03.24 00:00.00 000000'),stringdate('1995.04.21 00:00.00 000000'),stringdate('1995.03.29 00:00.00 000000'),2,25.360000,'Around the Horn','Brook Farm\15\12Stratford St. Mary','Colchester','Essex','CO7 6JX','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10454,'LAMAI',4,stringdate('1995.03.24 00:00.00 000000'),stringdate('1995.04.21 00:00.00 000000'),stringdate('1995.03.28 00:00.00 000000'),3,2.740000,'La maison d\47Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10455,'WARTH',8,stringdate('1995.03.27 00:00.00 000000'),stringdate('1995.05.08 00:00.00 000000'),stringdate('1995.04.03 00:00.00 000000'),2,180.450000,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10456,'KOENE',8,stringdate('1995.03.28 00:00.00 000000'),stringdate('1995.05.09 00:00.00 000000'),stringdate('1995.03.31 00:00.00 000000'),2,8.120000,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10457,'KOENE',2,stringdate('1995.03.28 00:00.00 000000'),stringdate('1995.04.25 00:00.00 000000'),stringdate('1995.04.03 00:00.00 000000'),1,11.570000,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10458,'SUPRD',7,stringdate('1995.03.29 00:00.00 000000'),stringdate('1995.04.26 00:00.00 000000'),stringdate('1995.04.04 00:00.00 000000'),3,147.060000,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10459,'VICTE',4,stringdate('1995.03.30 00:00.00 000000'),stringdate('1995.04.27 00:00.00 000000'),stringdate('1995.03.31 00:00.00 000000'),2,25.090000,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10460,'FOLKO',8,stringdate('1995.03.31 00:00.00 000000'),stringdate('1995.04.28 00:00.00 000000'),stringdate('1995.04.03 00:00.00 000000'),1,16.270000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10461,'LILAS',1,stringdate('1995.03.31 00:00.00 000000'),stringdate('1995.04.28 00:00.00 000000'),stringdate('1995.04.05 00:00.00 000000'),3,148.610000,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10462,'CONSH',2,stringdate('1995.04.03 00:00.00 000000'),stringdate('1995.05.01 00:00.00 000000'),stringdate('1995.04.18 00:00.00 000000'),1,6.170000,'Consolidated Holdings','Berkeley Gardens\15\1212  Brewery ','London',NULL,'WX1 6LT','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10463,'SUPRD',5,stringdate('1995.04.04 00:00.00 000000'),stringdate('1995.05.02 00:00.00 000000'),stringdate('1995.04.06 00:00.00 000000'),3,14.780000,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10464,'FURIB',4,stringdate('1995.04.04 00:00.00 000000'),stringdate('1995.05.02 00:00.00 000000'),stringdate('1995.04.14 00:00.00 000000'),2,89.000000,'Furia Bacalhau e Frutos do Mar','Jardim das rosas n. 32','Lisboa',NULL,'1675','Portugal');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10465,'VAFFE',1,stringdate('1995.04.05 00:00.00 000000'),stringdate('1995.05.03 00:00.00 000000'),stringdate('1995.04.14 00:00.00 000000'),3,145.040000,'Vaffeljernet','Smagslget 45','rhus',NULL,'8200','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10466,'COMMI',4,stringdate('1995.04.06 00:00.00 000000'),stringdate('1995.05.04 00:00.00 000000'),stringdate('1995.04.13 00:00.00 000000'),1,11.930000,'Comrcio Mineiro','Av. dos Lusadas, 23','So Paulo','SP','05432-043','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10467,'MAGAA',8,stringdate('1995.04.06 00:00.00 000000'),stringdate('1995.05.04 00:00.00 000000'),stringdate('1995.04.11 00:00.00 000000'),2,4.930000,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10468,'KOENE',3,stringdate('1995.04.07 00:00.00 000000'),stringdate('1995.05.05 00:00.00 000000'),stringdate('1995.04.12 00:00.00 000000'),3,44.120000,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10469,'WHITC',1,stringdate('1995.04.10 00:00.00 000000'),stringdate('1995.05.08 00:00.00 000000'),stringdate('1995.04.14 00:00.00 000000'),1,60.180000,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10470,'BONAP',4,stringdate('1995.04.11 00:00.00 000000'),stringdate('1995.05.09 00:00.00 000000'),stringdate('1995.04.14 00:00.00 000000'),2,64.560000,'Bon app\47','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10471,'BSBEV',2,stringdate('1995.04.11 00:00.00 000000'),stringdate('1995.05.09 00:00.00 000000'),stringdate('1995.04.18 00:00.00 000000'),3,45.590000,'B\47s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10472,'SEVES',8,stringdate('1995.04.12 00:00.00 000000'),stringdate('1995.05.10 00:00.00 000000'),stringdate('1995.04.19 00:00.00 000000'),1,4.200000,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10473,'ISLAT',1,stringdate('1995.04.13 00:00.00 000000'),stringdate('1995.04.27 00:00.00 000000'),stringdate('1995.04.21 00:00.00 000000'),3,16.370000,'Island Trading','Garden House\15\12Crowther Way','Cowes','Isle of Wight','PO31 7PJ','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10474,'PERIC',5,stringdate('1995.04.13 00:00.00 000000'),stringdate('1995.05.11 00:00.00 000000'),stringdate('1995.04.21 00:00.00 000000'),2,83.490000,'Pericles Comidas clsicas','Calle Dr. Jorge Cash 321','Mxico D.F.',NULL,'05033','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10475,'SUPRD',9,stringdate('1995.04.14 00:00.00 000000'),stringdate('1995.05.12 00:00.00 000000'),stringdate('1995.05.05 00:00.00 000000'),1,68.520000,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10476,'HILAA',8,stringdate('1995.04.17 00:00.00 000000'),stringdate('1995.05.15 00:00.00 000000'),stringdate('1995.04.24 00:00.00 000000'),3,4.410000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10477,'PRINI',5,stringdate('1995.04.17 00:00.00 000000'),stringdate('1995.05.15 00:00.00 000000'),stringdate('1995.04.25 00:00.00 000000'),2,13.020000,'Princesa Isabel Vinhos','Estrada da sade n. 58','Lisboa',NULL,'1756','Portugal');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10478,'VICTE',2,stringdate('1995.04.18 00:00.00 000000'),stringdate('1995.05.02 00:00.00 000000'),stringdate('1995.04.26 00:00.00 000000'),3,4.810000,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10479,'RATTC',3,stringdate('1995.04.19 00:00.00 000000'),stringdate('1995.05.17 00:00.00 000000'),stringdate('1995.04.21 00:00.00 000000'),3,708.950000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10480,'FOLIG',6,stringdate('1995.04.20 00:00.00 000000'),stringdate('1995.05.18 00:00.00 000000'),stringdate('1995.04.24 00:00.00 000000'),2,1.350000,'Folies gourmandes','184, chausse de Tournai','Lille',NULL,'59000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10481,'RICAR',8,stringdate('1995.04.20 00:00.00 000000'),stringdate('1995.05.18 00:00.00 000000'),stringdate('1995.04.25 00:00.00 000000'),2,64.330000,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10482,'LAZYK',1,stringdate('1995.04.21 00:00.00 000000'),stringdate('1995.05.19 00:00.00 000000'),stringdate('1995.05.11 00:00.00 000000'),3,7.480000,'Lazy K Kountry Store','12 Orchestra Terrace','Walla Walla','WA','99362','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10483,'WHITC',7,stringdate('1995.04.24 00:00.00 000000'),stringdate('1995.05.22 00:00.00 000000'),stringdate('1995.05.26 00:00.00 000000'),2,15.280000,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10484,'BSBEV',3,stringdate('1995.04.24 00:00.00 000000'),stringdate('1995.05.22 00:00.00 000000'),stringdate('1995.05.02 00:00.00 000000'),3,6.880000,'B\47s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10485,'LINOD',4,stringdate('1995.04.25 00:00.00 000000'),stringdate('1995.05.09 00:00.00 000000'),stringdate('1995.05.01 00:00.00 000000'),2,64.450000,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10486,'HILAA',1,stringdate('1995.04.26 00:00.00 000000'),stringdate('1995.05.24 00:00.00 000000'),stringdate('1995.05.03 00:00.00 000000'),2,30.530000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10487,'QUEEN',2,stringdate('1995.04.26 00:00.00 000000'),stringdate('1995.05.24 00:00.00 000000'),stringdate('1995.04.28 00:00.00 000000'),2,71.070000,'Queen Cozinha','Alameda dos Canrios, 891','So Paulo','SP','05487-020','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10488,'FRANK',8,stringdate('1995.04.27 00:00.00 000000'),stringdate('1995.05.25 00:00.00 000000'),stringdate('1995.05.03 00:00.00 000000'),2,4.930000,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10489,'PICCO',6,stringdate('1995.04.28 00:00.00 000000'),stringdate('1995.05.26 00:00.00 000000'),stringdate('1995.05.10 00:00.00 000000'),2,5.290000,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10490,'HILAA',7,stringdate('1995.05.01 00:00.00 000000'),stringdate('1995.05.29 00:00.00 000000'),stringdate('1995.05.04 00:00.00 000000'),2,210.190000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10491,'FURIB',8,stringdate('1995.05.01 00:00.00 000000'),stringdate('1995.05.29 00:00.00 000000'),stringdate('1995.05.09 00:00.00 000000'),3,16.960000,'Furia Bacalhau e Frutos do Mar','Jardim das rosas n. 32','Lisboa',NULL,'1675','Portugal');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10492,'BOTTM',3,stringdate('1995.05.02 00:00.00 000000'),stringdate('1995.05.30 00:00.00 000000'),stringdate('1995.05.12 00:00.00 000000'),1,62.890000,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10493,'LAMAI',4,stringdate('1995.05.03 00:00.00 000000'),stringdate('1995.05.31 00:00.00 000000'),stringdate('1995.05.11 00:00.00 000000'),3,10.640000,'La maison d\47Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10494,'COMMI',4,stringdate('1995.05.03 00:00.00 000000'),stringdate('1995.05.31 00:00.00 000000'),stringdate('1995.05.10 00:00.00 000000'),2,65.990000,'Comrcio Mineiro','Av. dos Lusadas, 23','So Paulo','SP','05432-043','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10495,'LAUGB',3,stringdate('1995.05.04 00:00.00 000000'),stringdate('1995.06.01 00:00.00 000000'),stringdate('1995.05.12 00:00.00 000000'),3,4.650000,'Laughing Bacchus Wine Cellars','2319 Elm St.','Vancouver','BC','V3F 2K1','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10496,'TRADH',7,stringdate('1995.05.05 00:00.00 000000'),stringdate('1995.06.02 00:00.00 000000'),stringdate('1995.05.08 00:00.00 000000'),2,46.770000,'Tradio Hipermercados','Av. Ins de Castro, 414','So Paulo','SP','05634-030','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10497,'LEHMS',7,stringdate('1995.05.05 00:00.00 000000'),stringdate('1995.06.02 00:00.00 000000'),stringdate('1995.05.08 00:00.00 000000'),1,36.210000,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M. ',NULL,'60528','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10498,'HILAA',8,stringdate('1995.05.08 00:00.00 000000'),stringdate('1995.06.05 00:00.00 000000'),stringdate('1995.05.12 00:00.00 000000'),2,29.750000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10499,'LILAS',4,stringdate('1995.05.09 00:00.00 000000'),stringdate('1995.06.06 00:00.00 000000'),stringdate('1995.05.17 00:00.00 000000'),2,102.020000,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10500,'LAMAI',6,stringdate('1995.05.10 00:00.00 000000'),stringdate('1995.06.07 00:00.00 000000'),stringdate('1995.05.18 00:00.00 000000'),1,42.680000,'La maison d\47Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10501,'BLAUS',9,stringdate('1995.05.10 00:00.00 000000'),stringdate('1995.06.07 00:00.00 000000'),stringdate('1995.05.17 00:00.00 000000'),3,8.850000,'Blauer See Delikatessen','Forsterstr. 57','Mannheim',NULL,'68306','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10502,'PERIC',2,stringdate('1995.05.11 00:00.00 000000'),stringdate('1995.06.08 00:00.00 000000'),stringdate('1995.05.30 00:00.00 000000'),1,69.320000,'Pericles Comidas clsicas','Calle Dr. Jorge Cash 321','Mxico D.F.',NULL,'05033','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10503,'HUNGO',6,stringdate('1995.05.12 00:00.00 000000'),stringdate('1995.06.09 00:00.00 000000'),stringdate('1995.05.17 00:00.00 000000'),2,16.740000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10504,'WHITC',4,stringdate('1995.05.12 00:00.00 000000'),stringdate('1995.06.09 00:00.00 000000'),stringdate('1995.05.19 00:00.00 000000'),3,59.130000,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10505,'MEREP',3,stringdate('1995.05.15 00:00.00 000000'),stringdate('1995.06.12 00:00.00 000000'),stringdate('1995.05.22 00:00.00 000000'),3,7.130000,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10506,'KOENE',9,stringdate('1995.05.16 00:00.00 000000'),stringdate('1995.06.13 00:00.00 000000'),stringdate('1995.06.02 00:00.00 000000'),2,21.190000,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10507,'ANTON',7,stringdate('1995.05.16 00:00.00 000000'),stringdate('1995.06.13 00:00.00 000000'),stringdate('1995.05.23 00:00.00 000000'),1,47.450000,'Antonio Moreno Taquera','Mataderos  2312','Mxico D.F.',NULL,'05023','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10508,'OTTIK',1,stringdate('1995.05.17 00:00.00 000000'),stringdate('1995.06.14 00:00.00 000000'),stringdate('1995.06.13 00:00.00 000000'),2,4.990000,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10509,'BLAUS',4,stringdate('1995.05.18 00:00.00 000000'),stringdate('1995.06.15 00:00.00 000000'),stringdate('1995.05.30 00:00.00 000000'),1,0.150000,'Blauer See Delikatessen','Forsterstr. 57','Mannheim',NULL,'68306','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10510,'SAVEA',6,stringdate('1995.05.19 00:00.00 000000'),stringdate('1995.06.16 00:00.00 000000'),stringdate('1995.05.29 00:00.00 000000'),3,367.630000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10511,'BONAP',4,stringdate('1995.05.19 00:00.00 000000'),stringdate('1995.06.16 00:00.00 000000'),stringdate('1995.05.22 00:00.00 000000'),3,350.640000,'Bon app\47','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10512,'FAMIA',7,stringdate('1995.05.22 00:00.00 000000'),stringdate('1995.06.19 00:00.00 000000'),stringdate('1995.05.25 00:00.00 000000'),2,3.530000,'Familia Arquibaldo','Rua Ors, 92','So Paulo','SP','05442-030','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10513,'WANDK',7,stringdate('1995.05.23 00:00.00 000000'),stringdate('1995.07.04 00:00.00 000000'),stringdate('1995.05.29 00:00.00 000000'),1,105.650000,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10514,'ERNSH',3,stringdate('1995.05.23 00:00.00 000000'),stringdate('1995.06.20 00:00.00 000000'),stringdate('1995.06.16 00:00.00 000000'),2,789.950000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10515,'QUICK',2,stringdate('1995.05.24 00:00.00 000000'),stringdate('1995.06.07 00:00.00 000000'),stringdate('1995.06.23 00:00.00 000000'),1,204.470000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10516,'HUNGO',2,stringdate('1995.05.25 00:00.00 000000'),stringdate('1995.06.22 00:00.00 000000'),stringdate('1995.06.01 00:00.00 000000'),3,62.780000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10517,'NORTS',3,stringdate('1995.05.25 00:00.00 000000'),stringdate('1995.06.22 00:00.00 000000'),stringdate('1995.05.30 00:00.00 000000'),3,32.070000,'North/South','South House\15\12300 Queensbridge','London',NULL,'SW7 1RZ','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10518,'TORTU',4,stringdate('1995.05.26 00:00.00 000000'),stringdate('1995.06.09 00:00.00 000000'),stringdate('1995.06.05 00:00.00 000000'),2,218.150000,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'05033','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10519,'CHOPS',6,stringdate('1995.05.29 00:00.00 000000'),stringdate('1995.06.26 00:00.00 000000'),stringdate('1995.06.01 00:00.00 000000'),3,91.760000,'Chop-suey Chinese','Hauptstr. 31','Bern',NULL,'3012','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10520,'SANTG',7,stringdate('1995.05.30 00:00.00 000000'),stringdate('1995.06.27 00:00.00 000000'),stringdate('1995.06.01 00:00.00 000000'),1,13.370000,'Sant Gourmet','Erling Skakkes gate 78','Stavern',NULL,'4110','Norway');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10521,'CACTU',8,stringdate('1995.05.30 00:00.00 000000'),stringdate('1995.06.27 00:00.00 000000'),stringdate('1995.06.02 00:00.00 000000'),2,17.220000,'Cactus Comidas para llevar','Cerrito 333','Buenos Aires',NULL,'1010','Argentina');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10522,'LEHMS',4,stringdate('1995.05.31 00:00.00 000000'),stringdate('1995.06.28 00:00.00 000000'),stringdate('1995.06.06 00:00.00 000000'),1,45.330000,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M. ',NULL,'60528','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10523,'SEVES',7,stringdate('1995.06.01 00:00.00 000000'),stringdate('1995.06.29 00:00.00 000000'),stringdate('1995.06.30 00:00.00 000000'),2,77.630000,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10524,'BERGS',1,stringdate('1995.06.01 00:00.00 000000'),stringdate('1995.06.29 00:00.00 000000'),stringdate('1995.06.07 00:00.00 000000'),2,244.790000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10525,'BONAP',1,stringdate('1995.06.02 00:00.00 000000'),stringdate('1995.06.30 00:00.00 000000'),stringdate('1995.06.23 00:00.00 000000'),2,11.060000,'Bon app\47','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10526,'WARTH',4,stringdate('1995.06.05 00:00.00 000000'),stringdate('1995.07.03 00:00.00 000000'),stringdate('1995.06.15 00:00.00 000000'),2,58.590000,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10527,'QUICK',7,stringdate('1995.06.05 00:00.00 000000'),stringdate('1995.07.03 00:00.00 000000'),stringdate('1995.06.07 00:00.00 000000'),1,41.900000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10528,'GREAL',6,stringdate('1995.06.06 00:00.00 000000'),stringdate('1995.06.20 00:00.00 000000'),stringdate('1995.06.09 00:00.00 000000'),2,3.350000,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10529,'MAISD',5,stringdate('1995.06.07 00:00.00 000000'),stringdate('1995.07.05 00:00.00 000000'),stringdate('1995.06.09 00:00.00 000000'),2,66.690000,'Maison Dewey','Rue Joseph-Bens 532','Bruxelles',NULL,'B-1180','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10530,'PICCO',3,stringdate('1995.06.08 00:00.00 000000'),stringdate('1995.07.06 00:00.00 000000'),stringdate('1995.06.12 00:00.00 000000'),2,339.220000,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10531,'OCEAN',7,stringdate('1995.06.08 00:00.00 000000'),stringdate('1995.07.06 00:00.00 000000'),stringdate('1995.06.19 00:00.00 000000'),1,8.120000,'Ocano Atlntico Ltda.','Ing. Gustavo Moncada 8585\15\12Piso 20-A','Buenos Aires',NULL,'1010','Argentina');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10532,'EASTC',7,stringdate('1995.06.09 00:00.00 000000'),stringdate('1995.07.07 00:00.00 000000'),stringdate('1995.06.12 00:00.00 000000'),3,74.460000,'Eastern Connection','35 King George','London',NULL,'WX3 6FW','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10533,'FOLKO',8,stringdate('1995.06.12 00:00.00 000000'),stringdate('1995.07.10 00:00.00 000000'),stringdate('1995.06.22 00:00.00 000000'),1,188.040000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10534,'LEHMS',8,stringdate('1995.06.12 00:00.00 000000'),stringdate('1995.07.10 00:00.00 000000'),stringdate('1995.06.14 00:00.00 000000'),2,27.940000,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M. ',NULL,'60528','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10535,'ANTON',4,stringdate('1995.06.13 00:00.00 000000'),stringdate('1995.07.11 00:00.00 000000'),stringdate('1995.06.21 00:00.00 000000'),1,15.640000,'Antonio Moreno Taquera','Mataderos  2312','Mxico D.F.',NULL,'05023','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10536,'LEHMS',3,stringdate('1995.06.14 00:00.00 000000'),stringdate('1995.07.12 00:00.00 000000'),stringdate('1995.07.07 00:00.00 000000'),2,58.880000,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M. ',NULL,'60528','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10537,'RICSU',1,stringdate('1995.06.14 00:00.00 000000'),stringdate('1995.06.28 00:00.00 000000'),stringdate('1995.06.19 00:00.00 000000'),1,78.850000,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10538,'BSBEV',9,stringdate('1995.06.15 00:00.00 000000'),stringdate('1995.07.13 00:00.00 000000'),stringdate('1995.06.16 00:00.00 000000'),3,4.870000,'B\47s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10539,'BSBEV',6,stringdate('1995.06.16 00:00.00 000000'),stringdate('1995.07.14 00:00.00 000000'),stringdate('1995.06.23 00:00.00 000000'),3,12.360000,'B\47s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10540,'QUICK',3,stringdate('1995.06.19 00:00.00 000000'),stringdate('1995.07.17 00:00.00 000000'),stringdate('1995.07.14 00:00.00 000000'),3,1007.640000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10541,'HANAR',2,stringdate('1995.06.19 00:00.00 000000'),stringdate('1995.07.17 00:00.00 000000'),stringdate('1995.06.29 00:00.00 000000'),1,68.650000,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10542,'KOENE',1,stringdate('1995.06.20 00:00.00 000000'),stringdate('1995.07.18 00:00.00 000000'),stringdate('1995.06.26 00:00.00 000000'),3,10.950000,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10543,'LILAS',8,stringdate('1995.06.21 00:00.00 000000'),stringdate('1995.07.19 00:00.00 000000'),stringdate('1995.06.23 00:00.00 000000'),2,48.170000,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10544,'LONEP',4,stringdate('1995.06.21 00:00.00 000000'),stringdate('1995.07.19 00:00.00 000000'),stringdate('1995.06.30 00:00.00 000000'),1,24.910000,'Lonesome Pine Restaurant','89 Chiaroscuro Rd.','Portland','OR','97219','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10545,'LAZYK',8,stringdate('1995.06.22 00:00.00 000000'),stringdate('1995.07.20 00:00.00 000000'),stringdate('1995.07.27 00:00.00 000000'),2,11.920000,'Lazy K Kountry Store','12 Orchestra Terrace','Walla Walla','WA','99362','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10546,'VICTE',1,stringdate('1995.06.23 00:00.00 000000'),stringdate('1995.07.21 00:00.00 000000'),stringdate('1995.06.27 00:00.00 000000'),3,194.720000,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10547,'SEVES',3,stringdate('1995.06.23 00:00.00 000000'),stringdate('1995.07.21 00:00.00 000000'),stringdate('1995.07.03 00:00.00 000000'),2,178.430000,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10548,'TOMSP',3,stringdate('1995.06.26 00:00.00 000000'),stringdate('1995.07.24 00:00.00 000000'),stringdate('1995.07.03 00:00.00 000000'),2,1.430000,'Toms Spezialitten','Luisenstr. 48','Mnster',NULL,'44087','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10549,'QUICK',5,stringdate('1995.06.27 00:00.00 000000'),stringdate('1995.07.11 00:00.00 000000'),stringdate('1995.06.30 00:00.00 000000'),1,171.240000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10550,'GODOS',7,stringdate('1995.06.28 00:00.00 000000'),stringdate('1995.07.26 00:00.00 000000'),stringdate('1995.07.07 00:00.00 000000'),3,4.320000,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10551,'FURIB',4,stringdate('1995.06.28 00:00.00 000000'),stringdate('1995.08.09 00:00.00 000000'),stringdate('1995.07.07 00:00.00 000000'),3,72.950000,'Furia Bacalhau e Frutos do Mar','Jardim das rosas n. 32','Lisboa',NULL,'1675','Portugal');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10552,'HILAA',2,stringdate('1995.06.29 00:00.00 000000'),stringdate('1995.07.27 00:00.00 000000'),stringdate('1995.07.06 00:00.00 000000'),1,83.220000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10553,'WARTH',2,stringdate('1995.06.30 00:00.00 000000'),stringdate('1995.07.28 00:00.00 000000'),stringdate('1995.07.04 00:00.00 000000'),2,149.490000,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10554,'OTTIK',4,stringdate('1995.06.30 00:00.00 000000'),stringdate('1995.07.28 00:00.00 000000'),stringdate('1995.07.06 00:00.00 000000'),3,120.970000,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10555,'SAVEA',6,stringdate('1995.07.03 00:00.00 000000'),stringdate('1995.07.31 00:00.00 000000'),stringdate('1995.07.05 00:00.00 000000'),3,252.490000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10556,'SIMOB',2,stringdate('1995.07.04 00:00.00 000000'),stringdate('1995.08.15 00:00.00 000000'),stringdate('1995.07.14 00:00.00 000000'),1,9.800000,'Simons bistro','Vinbltet 34','Kbenhavn',NULL,'1734','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10557,'LEHMS',9,stringdate('1995.07.04 00:00.00 000000'),stringdate('1995.07.18 00:00.00 000000'),stringdate('1995.07.07 00:00.00 000000'),2,96.720000,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M. ',NULL,'60528','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10558,'AROUT',1,stringdate('1995.07.05 00:00.00 000000'),stringdate('1995.08.02 00:00.00 000000'),stringdate('1995.07.11 00:00.00 000000'),2,72.970000,'Around the Horn','Brook Farm\15\12Stratford St. Mary','Colchester','Essex','CO7 6JX','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10559,'BLONP',6,stringdate('1995.07.06 00:00.00 000000'),stringdate('1995.08.03 00:00.00 000000'),stringdate('1995.07.14 00:00.00 000000'),1,8.050000,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10560,'FRANK',8,stringdate('1995.07.07 00:00.00 000000'),stringdate('1995.08.04 00:00.00 000000'),stringdate('1995.07.10 00:00.00 000000'),1,36.650000,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10561,'FOLKO',2,stringdate('1995.07.07 00:00.00 000000'),stringdate('1995.08.04 00:00.00 000000'),stringdate('1995.07.10 00:00.00 000000'),2,242.210000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10562,'REGGC',1,stringdate('1995.07.10 00:00.00 000000'),stringdate('1995.08.07 00:00.00 000000'),stringdate('1995.07.13 00:00.00 000000'),1,22.950000,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10563,'RICAR',2,stringdate('1995.07.11 00:00.00 000000'),stringdate('1995.08.22 00:00.00 000000'),stringdate('1995.07.25 00:00.00 000000'),2,60.430000,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10564,'RATTC',4,stringdate('1995.07.11 00:00.00 000000'),stringdate('1995.08.08 00:00.00 000000'),stringdate('1995.07.17 00:00.00 000000'),3,13.750000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10565,'MEREP',8,stringdate('1995.07.12 00:00.00 000000'),stringdate('1995.08.09 00:00.00 000000'),stringdate('1995.07.19 00:00.00 000000'),2,7.150000,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10566,'BLONP',9,stringdate('1995.07.13 00:00.00 000000'),stringdate('1995.08.10 00:00.00 000000'),stringdate('1995.07.19 00:00.00 000000'),1,88.400000,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10567,'HUNGO',1,stringdate('1995.07.13 00:00.00 000000'),stringdate('1995.08.10 00:00.00 000000'),stringdate('1995.07.18 00:00.00 000000'),1,33.970000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10568,'GALED',3,stringdate('1995.07.14 00:00.00 000000'),stringdate('1995.08.11 00:00.00 000000'),stringdate('1995.08.09 00:00.00 000000'),3,6.540000,'Galera del gastronmo','Rambla de Catalua, 23','Barcelona',NULL,'8022','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10569,'RATTC',5,stringdate('1995.07.17 00:00.00 000000'),stringdate('1995.08.14 00:00.00 000000'),stringdate('1995.08.11 00:00.00 000000'),1,58.980000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10570,'MEREP',3,stringdate('1995.07.18 00:00.00 000000'),stringdate('1995.08.15 00:00.00 000000'),stringdate('1995.07.20 00:00.00 000000'),3,188.990000,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10571,'ERNSH',8,stringdate('1995.07.18 00:00.00 000000'),stringdate('1995.08.29 00:00.00 000000'),stringdate('1995.08.04 00:00.00 000000'),3,26.060000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10572,'BERGS',3,stringdate('1995.07.19 00:00.00 000000'),stringdate('1995.08.16 00:00.00 000000'),stringdate('1995.07.26 00:00.00 000000'),2,116.430000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10573,'ANTON',7,stringdate('1995.07.20 00:00.00 000000'),stringdate('1995.08.17 00:00.00 000000'),stringdate('1995.07.21 00:00.00 000000'),3,84.840000,'Antonio Moreno Taquera','Mataderos  2312','Mxico D.F.',NULL,'05023','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10574,'TRAIH',4,stringdate('1995.07.20 00:00.00 000000'),stringdate('1995.08.17 00:00.00 000000'),stringdate('1995.07.31 00:00.00 000000'),2,37.600000,'Trail\47s Head Gourmet Provisioners','722 DaVinci Blvd.','Kirkland','WA','98034','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10575,'MORGK',5,stringdate('1995.07.21 00:00.00 000000'),stringdate('1995.08.04 00:00.00 000000'),stringdate('1995.07.31 00:00.00 000000'),1,127.340000,'Morgenstern Gesundkost','Heerstr. 22','Leipzig',NULL,'04179','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10576,'TORTU',3,stringdate('1995.07.24 00:00.00 000000'),stringdate('1995.08.07 00:00.00 000000'),stringdate('1995.07.31 00:00.00 000000'),3,18.560000,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'05033','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10577,'TRAIH',9,stringdate('1995.07.24 00:00.00 000000'),stringdate('1995.09.04 00:00.00 000000'),stringdate('1995.07.31 00:00.00 000000'),2,25.410000,'Trail\47s Head Gourmet Provisioners','722 DaVinci Blvd.','Kirkland','WA','98034','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10578,'BSBEV',4,stringdate('1995.07.25 00:00.00 000000'),stringdate('1995.08.22 00:00.00 000000'),stringdate('1995.08.25 00:00.00 000000'),3,29.600000,'B\47s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10579,'LETSS',1,stringdate('1995.07.26 00:00.00 000000'),stringdate('1995.08.23 00:00.00 000000'),stringdate('1995.08.04 00:00.00 000000'),2,13.730000,'Let\47s Stop N Shop','87 Polk St.\15\12Suite 5','San Francisco','CA','94117','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10580,'OTTIK',4,stringdate('1995.07.27 00:00.00 000000'),stringdate('1995.08.24 00:00.00 000000'),stringdate('1995.08.01 00:00.00 000000'),3,75.890000,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10581,'FAMIA',3,stringdate('1995.07.27 00:00.00 000000'),stringdate('1995.08.24 00:00.00 000000'),stringdate('1995.08.02 00:00.00 000000'),1,3.010000,'Familia Arquibaldo','Rua Ors, 92','So Paulo','SP','05442-030','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10582,'BLAUS',3,stringdate('1995.07.28 00:00.00 000000'),stringdate('1995.08.25 00:00.00 000000'),stringdate('1995.08.14 00:00.00 000000'),2,27.710000,'Blauer See Delikatessen','Forsterstr. 57','Mannheim',NULL,'68306','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10583,'WARTH',2,stringdate('1995.07.31 00:00.00 000000'),stringdate('1995.08.28 00:00.00 000000'),stringdate('1995.08.04 00:00.00 000000'),2,7.280000,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10584,'BLONP',4,stringdate('1995.07.31 00:00.00 000000'),stringdate('1995.08.28 00:00.00 000000'),stringdate('1995.08.04 00:00.00 000000'),1,59.140000,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10585,'WELLI',7,stringdate('1995.08.01 00:00.00 000000'),stringdate('1995.08.29 00:00.00 000000'),stringdate('1995.08.10 00:00.00 000000'),1,13.410000,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10586,'REGGC',9,stringdate('1995.08.02 00:00.00 000000'),stringdate('1995.08.30 00:00.00 000000'),stringdate('1995.08.09 00:00.00 000000'),1,0.480000,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10587,'QUEDE',1,stringdate('1995.08.02 00:00.00 000000'),stringdate('1995.08.30 00:00.00 000000'),stringdate('1995.08.09 00:00.00 000000'),1,62.520000,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10588,'QUICK',2,stringdate('1995.08.03 00:00.00 000000'),stringdate('1995.08.31 00:00.00 000000'),stringdate('1995.08.10 00:00.00 000000'),3,194.670000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10589,'GREAL',8,stringdate('1995.08.04 00:00.00 000000'),stringdate('1995.09.01 00:00.00 000000'),stringdate('1995.08.14 00:00.00 000000'),2,4.420000,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10590,'MEREP',4,stringdate('1995.08.07 00:00.00 000000'),stringdate('1995.09.04 00:00.00 000000'),stringdate('1995.08.14 00:00.00 000000'),3,44.770000,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10591,'VAFFE',1,stringdate('1995.08.07 00:00.00 000000'),stringdate('1995.08.21 00:00.00 000000'),stringdate('1995.08.16 00:00.00 000000'),1,55.920000,'Vaffeljernet','Smagslget 45','rhus',NULL,'8200','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10592,'LEHMS',3,stringdate('1995.08.08 00:00.00 000000'),stringdate('1995.09.05 00:00.00 000000'),stringdate('1995.08.16 00:00.00 000000'),1,32.100000,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M. ',NULL,'60528','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10593,'LEHMS',7,stringdate('1995.08.09 00:00.00 000000'),stringdate('1995.09.06 00:00.00 000000'),stringdate('1995.09.13 00:00.00 000000'),2,174.200000,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M. ',NULL,'60528','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10594,'OLDWO',3,stringdate('1995.08.09 00:00.00 000000'),stringdate('1995.09.06 00:00.00 000000'),stringdate('1995.08.16 00:00.00 000000'),2,5.240000,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10595,'ERNSH',2,stringdate('1995.08.10 00:00.00 000000'),stringdate('1995.09.07 00:00.00 000000'),stringdate('1995.08.14 00:00.00 000000'),1,96.780000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10596,'WHITC',8,stringdate('1995.08.11 00:00.00 000000'),stringdate('1995.09.08 00:00.00 000000'),stringdate('1995.09.12 00:00.00 000000'),1,16.340000,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10597,'PICCO',7,stringdate('1995.08.11 00:00.00 000000'),stringdate('1995.09.08 00:00.00 000000'),stringdate('1995.08.18 00:00.00 000000'),3,35.120000,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10598,'RATTC',1,stringdate('1995.08.14 00:00.00 000000'),stringdate('1995.09.11 00:00.00 000000'),stringdate('1995.08.18 00:00.00 000000'),3,44.420000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10599,'BSBEV',6,stringdate('1995.08.15 00:00.00 000000'),stringdate('1995.09.26 00:00.00 000000'),stringdate('1995.08.21 00:00.00 000000'),3,29.980000,'B\47s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10600,'HUNGC',4,stringdate('1995.08.16 00:00.00 000000'),stringdate('1995.09.13 00:00.00 000000'),stringdate('1995.08.21 00:00.00 000000'),1,45.130000,'Hungry Coyote Import Store','City Center Plaza\15\12516 Main St.','Elgin','OR','97827','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10601,'HILAA',7,stringdate('1995.08.16 00:00.00 000000'),stringdate('1995.09.27 00:00.00 000000'),stringdate('1995.08.22 00:00.00 000000'),1,58.300000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10602,'VAFFE',8,stringdate('1995.08.17 00:00.00 000000'),stringdate('1995.09.14 00:00.00 000000'),stringdate('1995.08.22 00:00.00 000000'),2,2.920000,'Vaffeljernet','Smagslget 45','rhus',NULL,'8200','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10603,'SAVEA',8,stringdate('1995.08.18 00:00.00 000000'),stringdate('1995.09.15 00:00.00 000000'),stringdate('1995.09.08 00:00.00 000000'),2,48.770000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10604,'FURIB',1,stringdate('1995.08.18 00:00.00 000000'),stringdate('1995.09.15 00:00.00 000000'),stringdate('1995.08.29 00:00.00 000000'),1,7.460000,'Furia Bacalhau e Frutos do Mar','Jardim das rosas n. 32','Lisboa',NULL,'1675','Portugal');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10605,'MEREP',1,stringdate('1995.08.21 00:00.00 000000'),stringdate('1995.09.18 00:00.00 000000'),stringdate('1995.08.29 00:00.00 000000'),2,379.130000,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10606,'TRADH',4,stringdate('1995.08.22 00:00.00 000000'),stringdate('1995.09.19 00:00.00 000000'),stringdate('1995.08.31 00:00.00 000000'),3,79.400000,'Tradio Hipermercados','Av. Ins de Castro, 414','So Paulo','SP','05634-030','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10607,'SAVEA',5,stringdate('1995.08.22 00:00.00 000000'),stringdate('1995.09.19 00:00.00 000000'),stringdate('1995.08.25 00:00.00 000000'),1,200.240000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10608,'TOMSP',4,stringdate('1995.08.23 00:00.00 000000'),stringdate('1995.09.20 00:00.00 000000'),stringdate('1995.09.01 00:00.00 000000'),2,27.790000,'Toms Spezialitten','Luisenstr. 48','Mnster',NULL,'44087','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10609,'DUMON',7,stringdate('1995.08.24 00:00.00 000000'),stringdate('1995.09.21 00:00.00 000000'),stringdate('1995.08.30 00:00.00 000000'),2,1.850000,'Du monde entier','67, rue des Cinquante Otages','Nantes',NULL,'44000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10610,'LAMAI',8,stringdate('1995.08.25 00:00.00 000000'),stringdate('1995.09.22 00:00.00 000000'),stringdate('1995.09.06 00:00.00 000000'),1,26.780000,'La maison d\47Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10611,'WOLZA',6,stringdate('1995.08.25 00:00.00 000000'),stringdate('1995.09.22 00:00.00 000000'),stringdate('1995.09.01 00:00.00 000000'),2,80.650000,'Wolski Zajazd','ul. Filtrowa 68','Warszawa',NULL,'01-012','Poland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10612,'SAVEA',1,stringdate('1995.08.28 00:00.00 000000'),stringdate('1995.09.25 00:00.00 000000'),stringdate('1995.09.01 00:00.00 000000'),2,544.080000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10613,'HILAA',4,stringdate('1995.08.29 00:00.00 000000'),stringdate('1995.09.26 00:00.00 000000'),stringdate('1995.09.01 00:00.00 000000'),2,8.110000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10614,'BLAUS',8,stringdate('1995.08.29 00:00.00 000000'),stringdate('1995.09.26 00:00.00 000000'),stringdate('1995.09.01 00:00.00 000000'),3,1.930000,'Blauer See Delikatessen','Forsterstr. 57','Mannheim',NULL,'68306','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10615,'WILMK',2,stringdate('1995.08.30 00:00.00 000000'),stringdate('1995.09.27 00:00.00 000000'),stringdate('1995.09.06 00:00.00 000000'),3,0.750000,'Wilman Kala','Keskuskatu 45','Helsinki',NULL,'21240','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10616,'GREAL',1,stringdate('1995.08.31 00:00.00 000000'),stringdate('1995.09.28 00:00.00 000000'),stringdate('1995.09.05 00:00.00 000000'),2,116.530000,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10617,'GREAL',4,stringdate('1995.08.31 00:00.00 000000'),stringdate('1995.09.28 00:00.00 000000'),stringdate('1995.09.04 00:00.00 000000'),2,18.530000,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10618,'MEREP',1,stringdate('1995.09.01 00:00.00 000000'),stringdate('1995.10.13 00:00.00 000000'),stringdate('1995.09.08 00:00.00 000000'),1,154.680000,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10619,'MEREP',3,stringdate('1995.09.04 00:00.00 000000'),stringdate('1995.10.02 00:00.00 000000'),stringdate('1995.09.07 00:00.00 000000'),3,91.050000,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10620,'LAUGB',2,stringdate('1995.09.05 00:00.00 000000'),stringdate('1995.10.03 00:00.00 000000'),stringdate('1995.09.14 00:00.00 000000'),3,0.940000,'Laughing Bacchus Wine Cellars','2319 Elm St.','Vancouver','BC','V3F 2K1','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10621,'ISLAT',4,stringdate('1995.09.05 00:00.00 000000'),stringdate('1995.10.03 00:00.00 000000'),stringdate('1995.09.11 00:00.00 000000'),2,23.730000,'Island Trading','Garden House\15\12Crowther Way','Cowes','Isle of Wight','PO31 7PJ','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10622,'RICAR',4,stringdate('1995.09.06 00:00.00 000000'),stringdate('1995.10.04 00:00.00 000000'),stringdate('1995.09.11 00:00.00 000000'),3,50.970000,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10623,'FRANK',8,stringdate('1995.09.07 00:00.00 000000'),stringdate('1995.10.05 00:00.00 000000'),stringdate('1995.09.12 00:00.00 000000'),2,97.180000,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10624,'THECR',4,stringdate('1995.09.07 00:00.00 000000'),stringdate('1995.10.05 00:00.00 000000'),stringdate('1995.09.19 00:00.00 000000'),2,94.800000,'The Cracker Box','55 Grizzly Peak Rd.','Butte','MT','59801','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10625,'ANATR',3,stringdate('1995.09.08 00:00.00 000000'),stringdate('1995.10.06 00:00.00 000000'),stringdate('1995.09.14 00:00.00 000000'),1,43.900000,'Ana Trujillo Emparedados y helados','Avda. de la Constitucin 2222','Mxico D.F.',NULL,'05021','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10626,'BERGS',1,stringdate('1995.09.11 00:00.00 000000'),stringdate('1995.10.09 00:00.00 000000'),stringdate('1995.09.20 00:00.00 000000'),2,138.690000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10627,'SAVEA',8,stringdate('1995.09.11 00:00.00 000000'),stringdate('1995.10.23 00:00.00 000000'),stringdate('1995.09.21 00:00.00 000000'),3,107.460000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10628,'BLONP',4,stringdate('1995.09.12 00:00.00 000000'),stringdate('1995.10.10 00:00.00 000000'),stringdate('1995.09.20 00:00.00 000000'),3,30.360000,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10629,'GODOS',4,stringdate('1995.09.12 00:00.00 000000'),stringdate('1995.10.10 00:00.00 000000'),stringdate('1995.09.20 00:00.00 000000'),3,85.460000,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10630,'KOENE',1,stringdate('1995.09.13 00:00.00 000000'),stringdate('1995.10.11 00:00.00 000000'),stringdate('1995.09.19 00:00.00 000000'),2,32.350000,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10631,'LAMAI',8,stringdate('1995.09.14 00:00.00 000000'),stringdate('1995.10.12 00:00.00 000000'),stringdate('1995.09.15 00:00.00 000000'),1,0.870000,'La maison d\47Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10632,'WANDK',8,stringdate('1995.09.14 00:00.00 000000'),stringdate('1995.10.12 00:00.00 000000'),stringdate('1995.09.19 00:00.00 000000'),1,41.380000,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10633,'ERNSH',7,stringdate('1995.09.15 00:00.00 000000'),stringdate('1995.10.13 00:00.00 000000'),stringdate('1995.09.18 00:00.00 000000'),3,477.900000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10634,'FOLIG',4,stringdate('1995.09.15 00:00.00 000000'),stringdate('1995.10.13 00:00.00 000000'),stringdate('1995.09.21 00:00.00 000000'),3,487.380000,'Folies gourmandes','184, chausse de Tournai','Lille',NULL,'59000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10635,'MAGAA',8,stringdate('1995.09.18 00:00.00 000000'),stringdate('1995.10.16 00:00.00 000000'),stringdate('1995.09.21 00:00.00 000000'),3,47.460000,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10636,'WARTH',4,stringdate('1995.09.19 00:00.00 000000'),stringdate('1995.10.17 00:00.00 000000'),stringdate('1995.09.26 00:00.00 000000'),1,1.150000,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10637,'QUEEN',6,stringdate('1995.09.19 00:00.00 000000'),stringdate('1995.10.17 00:00.00 000000'),stringdate('1995.09.26 00:00.00 000000'),1,201.290000,'Queen Cozinha','Alameda dos Canrios, 891','So Paulo','SP','05487-020','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10638,'LINOD',3,stringdate('1995.09.20 00:00.00 000000'),stringdate('1995.10.18 00:00.00 000000'),stringdate('1995.10.02 00:00.00 000000'),1,158.440000,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10639,'SANTG',7,stringdate('1995.09.20 00:00.00 000000'),stringdate('1995.10.18 00:00.00 000000'),stringdate('1995.09.27 00:00.00 000000'),3,38.640000,'Sant Gourmet','Erling Skakkes gate 78','Stavern',NULL,'4110','Norway');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10640,'WANDK',4,stringdate('1995.09.21 00:00.00 000000'),stringdate('1995.10.19 00:00.00 000000'),stringdate('1995.09.28 00:00.00 000000'),1,23.550000,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10641,'HILAA',4,stringdate('1995.09.22 00:00.00 000000'),stringdate('1995.10.20 00:00.00 000000'),stringdate('1995.09.26 00:00.00 000000'),2,179.610000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10642,'SIMOB',7,stringdate('1995.09.22 00:00.00 000000'),stringdate('1995.10.20 00:00.00 000000'),stringdate('1995.10.06 00:00.00 000000'),3,41.890000,'Simons bistro','Vinbltet 34','Kbenhavn',NULL,'1734','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10643,'ALFKI',6,stringdate('1995.09.25 00:00.00 000000'),stringdate('1995.10.23 00:00.00 000000'),stringdate('1995.10.03 00:00.00 000000'),1,29.460000,'Alfreds Futterkiste','Obere Str. 57','Berlin',NULL,'12209','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10644,'WELLI',3,stringdate('1995.09.25 00:00.00 000000'),stringdate('1995.10.23 00:00.00 000000'),stringdate('1995.10.02 00:00.00 000000'),2,0.140000,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10645,'HANAR',4,stringdate('1995.09.26 00:00.00 000000'),stringdate('1995.10.24 00:00.00 000000'),stringdate('1995.10.03 00:00.00 000000'),1,12.410000,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10646,'HUNGO',9,stringdate('1995.09.27 00:00.00 000000'),stringdate('1995.11.08 00:00.00 000000'),stringdate('1995.10.04 00:00.00 000000'),3,142.330000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10647,'QUEDE',4,stringdate('1995.09.27 00:00.00 000000'),stringdate('1995.10.11 00:00.00 000000'),stringdate('1995.10.04 00:00.00 000000'),2,45.540000,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10648,'RICAR',5,stringdate('1995.09.28 00:00.00 000000'),stringdate('1995.11.09 00:00.00 000000'),stringdate('1995.10.10 00:00.00 000000'),2,14.250000,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10649,'MAISD',5,stringdate('1995.09.28 00:00.00 000000'),stringdate('1995.10.26 00:00.00 000000'),stringdate('1995.09.29 00:00.00 000000'),3,6.200000,'Maison Dewey','Rue Joseph-Bens 532','Bruxelles',NULL,'B-1180','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10650,'FAMIA',5,stringdate('1995.09.29 00:00.00 000000'),stringdate('1995.10.27 00:00.00 000000'),stringdate('1995.10.04 00:00.00 000000'),3,176.810000,'Familia Arquibaldo','Rua Ors, 92','So Paulo','SP','05442-030','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10651,'WANDK',8,stringdate('1995.10.02 00:00.00 000000'),stringdate('1995.10.30 00:00.00 000000'),stringdate('1995.10.12 00:00.00 000000'),2,20.600000,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10652,'GOURL',4,stringdate('1995.10.02 00:00.00 000000'),stringdate('1995.10.30 00:00.00 000000'),stringdate('1995.10.09 00:00.00 000000'),2,7.140000,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10653,'FRANK',1,stringdate('1995.10.03 00:00.00 000000'),stringdate('1995.10.31 00:00.00 000000'),stringdate('1995.10.20 00:00.00 000000'),1,93.250000,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10654,'BERGS',5,stringdate('1995.10.03 00:00.00 000000'),stringdate('1995.10.31 00:00.00 000000'),stringdate('1995.10.12 00:00.00 000000'),1,55.260000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10655,'REGGC',1,stringdate('1995.10.04 00:00.00 000000'),stringdate('1995.11.01 00:00.00 000000'),stringdate('1995.10.12 00:00.00 000000'),2,4.410000,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10656,'GREAL',6,stringdate('1995.10.05 00:00.00 000000'),stringdate('1995.11.02 00:00.00 000000'),stringdate('1995.10.11 00:00.00 000000'),1,57.150000,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10657,'SAVEA',2,stringdate('1995.10.05 00:00.00 000000'),stringdate('1995.11.02 00:00.00 000000'),stringdate('1995.10.16 00:00.00 000000'),2,352.690000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10658,'QUICK',4,stringdate('1995.10.06 00:00.00 000000'),stringdate('1995.11.03 00:00.00 000000'),stringdate('1995.10.09 00:00.00 000000'),1,364.150000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10659,'QUEEN',7,stringdate('1995.10.06 00:00.00 000000'),stringdate('1995.11.03 00:00.00 000000'),stringdate('1995.10.11 00:00.00 000000'),2,105.810000,'Queen Cozinha','Alameda dos Canrios, 891','So Paulo','SP','05487-020','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10660,'HUNGC',8,stringdate('1995.10.09 00:00.00 000000'),stringdate('1995.11.06 00:00.00 000000'),stringdate('1995.11.15 00:00.00 000000'),1,111.290000,'Hungry Coyote Import Store','City Center Plaza\15\12516 Main St.','Elgin','OR','97827','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10661,'HUNGO',7,stringdate('1995.10.10 00:00.00 000000'),stringdate('1995.11.07 00:00.00 000000'),stringdate('1995.10.16 00:00.00 000000'),3,17.550000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10662,'LONEP',3,stringdate('1995.10.10 00:00.00 000000'),stringdate('1995.11.07 00:00.00 000000'),stringdate('1995.10.19 00:00.00 000000'),2,1.280000,'Lonesome Pine Restaurant','89 Chiaroscuro Rd.','Portland','OR','97219','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10663,'BONAP',2,stringdate('1995.10.11 00:00.00 000000'),stringdate('1995.10.25 00:00.00 000000'),stringdate('1995.11.03 00:00.00 000000'),2,113.150000,'Bon app\47','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10664,'FURIB',1,stringdate('1995.10.11 00:00.00 000000'),stringdate('1995.11.08 00:00.00 000000'),stringdate('1995.10.20 00:00.00 000000'),3,1.270000,'Furia Bacalhau e Frutos do Mar','Jardim das rosas n. 32','Lisboa',NULL,'1675','Portugal');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10665,'LONEP',1,stringdate('1995.10.12 00:00.00 000000'),stringdate('1995.11.09 00:00.00 000000'),stringdate('1995.10.18 00:00.00 000000'),2,26.310000,'Lonesome Pine Restaurant','89 Chiaroscuro Rd.','Portland','OR','97219','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10666,'RICSU',7,stringdate('1995.10.13 00:00.00 000000'),stringdate('1995.11.10 00:00.00 000000'),stringdate('1995.10.23 00:00.00 000000'),2,232.420000,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10667,'ERNSH',7,stringdate('1995.10.13 00:00.00 000000'),stringdate('1995.11.10 00:00.00 000000'),stringdate('1995.10.20 00:00.00 000000'),1,78.090000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10668,'WANDK',1,stringdate('1995.10.16 00:00.00 000000'),stringdate('1995.11.13 00:00.00 000000'),stringdate('1995.10.24 00:00.00 000000'),2,47.220000,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10669,'SIMOB',2,stringdate('1995.10.16 00:00.00 000000'),stringdate('1995.11.13 00:00.00 000000'),stringdate('1995.10.23 00:00.00 000000'),1,24.390000,'Simons bistro','Vinbltet 34','Kbenhavn',NULL,'1734','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10670,'FRANK',4,stringdate('1995.10.17 00:00.00 000000'),stringdate('1995.11.14 00:00.00 000000'),stringdate('1995.10.19 00:00.00 000000'),1,203.480000,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10671,'FRANR',1,stringdate('1995.10.18 00:00.00 000000'),stringdate('1995.11.15 00:00.00 000000'),stringdate('1995.10.25 00:00.00 000000'),1,30.340000,'France restauration','54, rue Royale','Nantes',NULL,'44000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10672,'BERGS',9,stringdate('1995.10.18 00:00.00 000000'),stringdate('1995.11.01 00:00.00 000000'),stringdate('1995.10.27 00:00.00 000000'),2,95.750000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10673,'WILMK',2,stringdate('1995.10.19 00:00.00 000000'),stringdate('1995.11.16 00:00.00 000000'),stringdate('1995.10.20 00:00.00 000000'),1,22.760000,'Wilman Kala','Keskuskatu 45','Helsinki',NULL,'21240','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10674,'ISLAT',4,stringdate('1995.10.19 00:00.00 000000'),stringdate('1995.11.16 00:00.00 000000'),stringdate('1995.10.31 00:00.00 000000'),2,0.900000,'Island Trading','Garden House\15\12Crowther Way','Cowes','Isle of Wight','PO31 7PJ','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10675,'FRANK',5,stringdate('1995.10.20 00:00.00 000000'),stringdate('1995.11.17 00:00.00 000000'),stringdate('1995.10.24 00:00.00 000000'),2,31.850000,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10676,'TORTU',2,stringdate('1995.10.23 00:00.00 000000'),stringdate('1995.11.20 00:00.00 000000'),stringdate('1995.10.30 00:00.00 000000'),2,2.010000,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'05033','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10677,'ANTON',1,stringdate('1995.10.23 00:00.00 000000'),stringdate('1995.11.20 00:00.00 000000'),stringdate('1995.10.27 00:00.00 000000'),3,4.030000,'Antonio Moreno Taquera','Mataderos  2312','Mxico D.F.',NULL,'05023','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10678,'SAVEA',7,stringdate('1995.10.24 00:00.00 000000'),stringdate('1995.11.21 00:00.00 000000'),stringdate('1995.11.16 00:00.00 000000'),3,388.980000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10679,'BLONP',8,stringdate('1995.10.24 00:00.00 000000'),stringdate('1995.11.21 00:00.00 000000'),stringdate('1995.10.31 00:00.00 000000'),3,27.940000,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10680,'OLDWO',1,stringdate('1995.10.25 00:00.00 000000'),stringdate('1995.11.22 00:00.00 000000'),stringdate('1995.10.27 00:00.00 000000'),1,26.610000,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10681,'GREAL',3,stringdate('1995.10.26 00:00.00 000000'),stringdate('1995.11.23 00:00.00 000000'),stringdate('1995.10.31 00:00.00 000000'),3,76.130000,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10682,'ANTON',3,stringdate('1995.10.26 00:00.00 000000'),stringdate('1995.11.23 00:00.00 000000'),stringdate('1995.11.01 00:00.00 000000'),2,36.130000,'Antonio Moreno Taquera','Mataderos  2312','Mxico D.F.',NULL,'05023','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10683,'DUMON',2,stringdate('1995.10.27 00:00.00 000000'),stringdate('1995.11.24 00:00.00 000000'),stringdate('1995.11.01 00:00.00 000000'),1,4.400000,'Du monde entier','67, rue des Cinquante Otages','Nantes',NULL,'44000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10684,'OTTIK',3,stringdate('1995.10.27 00:00.00 000000'),stringdate('1995.11.24 00:00.00 000000'),stringdate('1995.10.31 00:00.00 000000'),1,145.630000,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10685,'GOURL',4,stringdate('1995.10.30 00:00.00 000000'),stringdate('1995.11.13 00:00.00 000000'),stringdate('1995.11.03 00:00.00 000000'),2,33.750000,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10686,'PICCO',2,stringdate('1995.10.31 00:00.00 000000'),stringdate('1995.11.28 00:00.00 000000'),stringdate('1995.11.08 00:00.00 000000'),1,96.500000,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10687,'HUNGO',9,stringdate('1995.10.31 00:00.00 000000'),stringdate('1995.11.28 00:00.00 000000'),stringdate('1995.11.30 00:00.00 000000'),2,296.430000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10688,'VAFFE',4,stringdate('1995.11.01 00:00.00 000000'),stringdate('1995.11.15 00:00.00 000000'),stringdate('1995.11.07 00:00.00 000000'),2,299.090000,'Vaffeljernet','Smagslget 45','rhus',NULL,'8200','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10689,'BERGS',1,stringdate('1995.11.01 00:00.00 000000'),stringdate('1995.11.29 00:00.00 000000'),stringdate('1995.11.07 00:00.00 000000'),2,13.420000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10690,'HANAR',1,stringdate('1995.11.02 00:00.00 000000'),stringdate('1995.11.30 00:00.00 000000'),stringdate('1995.11.03 00:00.00 000000'),1,15.800000,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10691,'QUICK',2,stringdate('1995.11.03 00:00.00 000000'),stringdate('1995.12.15 00:00.00 000000'),stringdate('1995.11.22 00:00.00 000000'),2,810.050000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10692,'ALFKI',4,stringdate('1995.11.03 00:00.00 000000'),stringdate('1995.12.01 00:00.00 000000'),stringdate('1995.11.13 00:00.00 000000'),2,61.020000,'Alfred\47s Futterkiste','Obere Str. 57','Berlin',NULL,'12209','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10693,'WHITC',3,stringdate('1995.11.06 00:00.00 000000'),stringdate('1995.11.20 00:00.00 000000'),stringdate('1995.11.10 00:00.00 000000'),3,139.340000,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10694,'QUICK',8,stringdate('1995.11.06 00:00.00 000000'),stringdate('1995.12.04 00:00.00 000000'),stringdate('1995.11.09 00:00.00 000000'),3,398.360000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10695,'WILMK',7,stringdate('1995.11.07 00:00.00 000000'),stringdate('1995.12.19 00:00.00 000000'),stringdate('1995.11.14 00:00.00 000000'),1,16.720000,'Wilman Kala','Keskuskatu 45','Helsinki',NULL,'21240','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10696,'WHITC',8,stringdate('1995.11.08 00:00.00 000000'),stringdate('1995.12.20 00:00.00 000000'),stringdate('1995.11.14 00:00.00 000000'),3,102.550000,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10697,'LINOD',3,stringdate('1995.11.08 00:00.00 000000'),stringdate('1995.12.06 00:00.00 000000'),stringdate('1995.11.14 00:00.00 000000'),1,45.520000,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10698,'ERNSH',4,stringdate('1995.11.09 00:00.00 000000'),stringdate('1995.12.07 00:00.00 000000'),stringdate('1995.11.17 00:00.00 000000'),1,272.470000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10699,'MORGK',3,stringdate('1995.11.09 00:00.00 000000'),stringdate('1995.12.07 00:00.00 000000'),stringdate('1995.11.13 00:00.00 000000'),3,0.580000,'Morgenstern Gesundkost','Heerstr. 22','Leipzig',NULL,'04179','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10700,'SAVEA',3,stringdate('1995.11.10 00:00.00 000000'),stringdate('1995.12.08 00:00.00 000000'),stringdate('1995.11.16 00:00.00 000000'),1,65.100000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10701,'HUNGO',6,stringdate('1995.11.13 00:00.00 000000'),stringdate('1995.11.27 00:00.00 000000'),stringdate('1995.11.15 00:00.00 000000'),3,220.310000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10702,'ALFKI',4,stringdate('1995.11.13 00:00.00 000000'),stringdate('1995.12.25 00:00.00 000000'),stringdate('1995.11.21 00:00.00 000000'),1,23.940000,'Alfred\47s Futterkiste','Obere Str. 57','Berlin',NULL,'12209','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10703,'FOLKO',6,stringdate('1995.11.14 00:00.00 000000'),stringdate('1995.12.12 00:00.00 000000'),stringdate('1995.11.20 00:00.00 000000'),2,152.300000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10704,'QUEEN',6,stringdate('1995.11.14 00:00.00 000000'),stringdate('1995.12.12 00:00.00 000000'),stringdate('1995.12.08 00:00.00 000000'),1,4.780000,'Queen Cozinha','Alameda dos Canrios, 891','So Paulo','SP','05487-020','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10705,'HILAA',9,stringdate('1995.11.15 00:00.00 000000'),stringdate('1995.12.13 00:00.00 000000'),stringdate('1995.12.19 00:00.00 000000'),2,3.520000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10706,'OLDWO',8,stringdate('1995.11.16 00:00.00 000000'),stringdate('1995.12.14 00:00.00 000000'),stringdate('1995.11.21 00:00.00 000000'),3,135.630000,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10707,'AROUT',4,stringdate('1995.11.16 00:00.00 000000'),stringdate('1995.11.30 00:00.00 000000'),stringdate('1995.11.23 00:00.00 000000'),3,21.740000,'Around the Horn','Brook Farm\15\12Stratford St. Mary','Colchester','Essex','CO7 6JX','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10708,'THEBI',6,stringdate('1995.11.17 00:00.00 000000'),stringdate('1995.12.29 00:00.00 000000'),stringdate('1995.12.06 00:00.00 000000'),2,2.960000,'The Big Cheese','89 Jefferson Way\15\12Suite 2','Portland','OR','97201','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10709,'GOURL',1,stringdate('1995.11.17 00:00.00 000000'),stringdate('1995.12.15 00:00.00 000000'),stringdate('1995.12.21 00:00.00 000000'),3,210.800000,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10710,'FRANS',1,stringdate('1995.11.20 00:00.00 000000'),stringdate('1995.12.18 00:00.00 000000'),stringdate('1995.11.23 00:00.00 000000'),1,4.980000,'Franchi S.p.A.','Via Monte Bianco 34','Torino',NULL,'10100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10711,'SAVEA',5,stringdate('1995.11.21 00:00.00 000000'),stringdate('1996.01.02 00:00.00 000000'),stringdate('1995.11.29 00:00.00 000000'),2,52.410000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10712,'HUNGO',3,stringdate('1995.11.21 00:00.00 000000'),stringdate('1995.12.19 00:00.00 000000'),stringdate('1995.12.01 00:00.00 000000'),1,89.930000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10713,'SAVEA',1,stringdate('1995.11.22 00:00.00 000000'),stringdate('1995.12.20 00:00.00 000000'),stringdate('1995.11.24 00:00.00 000000'),1,167.050000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10714,'SAVEA',5,stringdate('1995.11.22 00:00.00 000000'),stringdate('1995.12.20 00:00.00 000000'),stringdate('1995.11.27 00:00.00 000000'),3,24.490000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10715,'BONAP',3,stringdate('1995.11.23 00:00.00 000000'),stringdate('1995.12.07 00:00.00 000000'),stringdate('1995.11.29 00:00.00 000000'),1,63.200000,'Bon app\47','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10716,'RANCH',4,stringdate('1995.11.24 00:00.00 000000'),stringdate('1995.12.22 00:00.00 000000'),stringdate('1995.11.27 00:00.00 000000'),2,22.570000,'Rancho grande','Av. del Libertador 900','Buenos Aires',NULL,'1010','Argentina');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10717,'FRANK',1,stringdate('1995.11.24 00:00.00 000000'),stringdate('1995.12.22 00:00.00 000000'),stringdate('1995.11.29 00:00.00 000000'),2,59.250000,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10718,'KOENE',1,stringdate('1995.11.27 00:00.00 000000'),stringdate('1995.12.25 00:00.00 000000'),stringdate('1995.11.29 00:00.00 000000'),3,170.880000,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10719,'LETSS',8,stringdate('1995.11.27 00:00.00 000000'),stringdate('1995.12.25 00:00.00 000000'),stringdate('1995.12.06 00:00.00 000000'),2,51.440000,'Let\47s Stop N Shop','87 Polk St.\15\12Suite 5','San Francisco','CA','94117','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10720,'QUEDE',8,stringdate('1995.11.28 00:00.00 000000'),stringdate('1995.12.12 00:00.00 000000'),stringdate('1995.12.06 00:00.00 000000'),2,9.530000,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10721,'QUICK',5,stringdate('1995.11.29 00:00.00 000000'),stringdate('1995.12.27 00:00.00 000000'),stringdate('1995.12.01 00:00.00 000000'),3,48.920000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10722,'SAVEA',8,stringdate('1995.11.29 00:00.00 000000'),stringdate('1996.01.10 00:00.00 000000'),stringdate('1995.12.05 00:00.00 000000'),1,74.580000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10723,'WHITC',3,stringdate('1995.11.30 00:00.00 000000'),stringdate('1995.12.28 00:00.00 000000'),stringdate('1995.12.26 00:00.00 000000'),1,21.720000,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10724,'MEREP',8,stringdate('1995.11.30 00:00.00 000000'),stringdate('1996.01.11 00:00.00 000000'),stringdate('1995.12.06 00:00.00 000000'),2,57.750000,'Mre Paillarde','43 rue St. Laurent','Montral','Qubec','H1J 1C3','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10725,'FAMIA',4,stringdate('1995.12.01 00:00.00 000000'),stringdate('1995.12.29 00:00.00 000000'),stringdate('1995.12.06 00:00.00 000000'),3,10.830000,'Familia Arquibaldo','Rua Ors, 92','So Paulo','SP','05442-030','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10726,'EASTC',4,stringdate('1995.12.04 00:00.00 000000'),stringdate('1995.12.18 00:00.00 000000'),stringdate('1996.01.05 00:00.00 000000'),1,16.560000,'Eastern Connection','35 King George','London',NULL,'WX3 6FW','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10727,'REGGC',2,stringdate('1995.12.04 00:00.00 000000'),stringdate('1996.01.01 00:00.00 000000'),stringdate('1996.01.05 00:00.00 000000'),1,89.900000,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10728,'QUEEN',4,stringdate('1995.12.05 00:00.00 000000'),stringdate('1996.01.02 00:00.00 000000'),stringdate('1995.12.12 00:00.00 000000'),2,58.330000,'Queen Cozinha','Alameda dos Canrios, 891','So Paulo','SP','05487-020','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10729,'LINOD',8,stringdate('1995.12.05 00:00.00 000000'),stringdate('1996.01.16 00:00.00 000000'),stringdate('1995.12.15 00:00.00 000000'),3,141.060000,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10730,'BONAP',5,stringdate('1995.12.06 00:00.00 000000'),stringdate('1996.01.03 00:00.00 000000'),stringdate('1995.12.15 00:00.00 000000'),1,20.120000,'Bon app\47','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10731,'CHOPS',7,stringdate('1995.12.07 00:00.00 000000'),stringdate('1996.01.04 00:00.00 000000'),stringdate('1995.12.15 00:00.00 000000'),1,96.650000,'Chop-suey Chinese','Hauptstr. 31','Bern',NULL,'3012','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10732,'BONAP',3,stringdate('1995.12.07 00:00.00 000000'),stringdate('1996.01.04 00:00.00 000000'),stringdate('1995.12.08 00:00.00 000000'),1,16.970000,'Bon app\47','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10733,'BERGS',1,stringdate('1995.12.08 00:00.00 000000'),stringdate('1996.01.05 00:00.00 000000'),stringdate('1995.12.11 00:00.00 000000'),3,110.110000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10734,'GOURL',2,stringdate('1995.12.08 00:00.00 000000'),stringdate('1996.01.05 00:00.00 000000'),stringdate('1995.12.13 00:00.00 000000'),3,1.630000,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10735,'LETSS',6,stringdate('1995.12.11 00:00.00 000000'),stringdate('1996.01.08 00:00.00 000000'),stringdate('1995.12.22 00:00.00 000000'),2,45.970000,'Let\47s Stop N Shop','87 Polk St.\15\12Suite 5','San Francisco','CA','94117','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10736,'HUNGO',9,stringdate('1995.12.12 00:00.00 000000'),stringdate('1996.01.09 00:00.00 000000'),stringdate('1995.12.22 00:00.00 000000'),2,44.100000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10737,'VINET',2,stringdate('1995.12.12 00:00.00 000000'),stringdate('1996.01.09 00:00.00 000000'),stringdate('1995.12.19 00:00.00 000000'),2,7.790000,'Vins et alcools Chevalier','59 rue de l\47Abbaye','Reims',NULL,'51100','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10738,'SPECD',2,stringdate('1995.12.13 00:00.00 000000'),stringdate('1996.01.10 00:00.00 000000'),stringdate('1995.12.19 00:00.00 000000'),1,2.910000,'Spcialits du monde','25, rue Lauriston','Paris',NULL,'75016','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10739,'VINET',3,stringdate('1995.12.13 00:00.00 000000'),stringdate('1996.01.10 00:00.00 000000'),stringdate('1995.12.18 00:00.00 000000'),3,11.080000,'Vins et alcools Chevalier','59 rue de l\47Abbaye','Reims',NULL,'51100','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10740,'WHITC',4,stringdate('1995.12.14 00:00.00 000000'),stringdate('1996.01.11 00:00.00 000000'),stringdate('1995.12.26 00:00.00 000000'),2,81.880000,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10741,'AROUT',4,stringdate('1995.12.15 00:00.00 000000'),stringdate('1995.12.29 00:00.00 000000'),stringdate('1995.12.19 00:00.00 000000'),3,10.960000,'Around the Horn','Brook Farm\15\12Stratford St. Mary','Colchester','Essex','CO7 6JX','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10742,'BOTTM',3,stringdate('1995.12.15 00:00.00 000000'),stringdate('1996.01.12 00:00.00 000000'),stringdate('1995.12.19 00:00.00 000000'),3,243.730000,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10743,'AROUT',1,stringdate('1995.12.18 00:00.00 000000'),stringdate('1996.01.15 00:00.00 000000'),stringdate('1995.12.22 00:00.00 000000'),2,23.720000,'Around the Horn','Brook Farm\15\12Stratford St. Mary','Colchester','Essex','CO7 6JX','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10744,'VAFFE',6,stringdate('1995.12.18 00:00.00 000000'),stringdate('1996.01.15 00:00.00 000000'),stringdate('1995.12.25 00:00.00 000000'),1,69.190000,'Vaffeljernet','Smagslget 45','rhus',NULL,'8200','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10745,'QUICK',9,stringdate('1995.12.19 00:00.00 000000'),stringdate('1996.01.16 00:00.00 000000'),stringdate('1995.12.28 00:00.00 000000'),1,3.520000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10746,'CHOPS',1,stringdate('1995.12.20 00:00.00 000000'),stringdate('1996.01.17 00:00.00 000000'),stringdate('1995.12.22 00:00.00 000000'),3,31.430000,'Chop-suey Chinese','Hauptstr. 31','Bern',NULL,'3012','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10747,'PICCO',6,stringdate('1995.12.20 00:00.00 000000'),stringdate('1996.01.17 00:00.00 000000'),stringdate('1995.12.27 00:00.00 000000'),1,117.330000,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10748,'SAVEA',3,stringdate('1995.12.21 00:00.00 000000'),stringdate('1996.01.18 00:00.00 000000'),stringdate('1995.12.29 00:00.00 000000'),1,232.550000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10749,'ISLAT',4,stringdate('1995.12.21 00:00.00 000000'),stringdate('1996.01.18 00:00.00 000000'),stringdate('1996.01.19 00:00.00 000000'),2,61.530000,'Island Trading','Garden House\15\12Crowther Way','Cowes','Isle of Wight','PO31 7PJ','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10750,'WARTH',9,stringdate('1995.12.22 00:00.00 000000'),stringdate('1996.01.19 00:00.00 000000'),stringdate('1995.12.25 00:00.00 000000'),1,79.300000,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10751,'RICSU',3,stringdate('1995.12.25 00:00.00 000000'),stringdate('1996.01.22 00:00.00 000000'),stringdate('1996.01.03 00:00.00 000000'),3,130.790000,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10752,'NORTS',2,stringdate('1995.12.25 00:00.00 000000'),stringdate('1996.01.22 00:00.00 000000'),stringdate('1995.12.29 00:00.00 000000'),3,1.390000,'North/South','South House\15\12300 Queensbridge','London',NULL,'SW7 1RZ','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10753,'FRANS',3,stringdate('1995.12.26 00:00.00 000000'),stringdate('1996.01.23 00:00.00 000000'),stringdate('1995.12.28 00:00.00 000000'),1,7.700000,'Franchi S.p.A.','Via Monte Bianco 34','Torino',NULL,'10100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10754,'MAGAA',6,stringdate('1995.12.26 00:00.00 000000'),stringdate('1996.01.23 00:00.00 000000'),stringdate('1995.12.28 00:00.00 000000'),3,2.380000,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10755,'BONAP',4,stringdate('1995.12.27 00:00.00 000000'),stringdate('1996.01.24 00:00.00 000000'),stringdate('1995.12.29 00:00.00 000000'),2,16.710000,'Bon app\47','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10756,'SPLIR',8,stringdate('1995.12.28 00:00.00 000000'),stringdate('1996.01.25 00:00.00 000000'),stringdate('1996.01.02 00:00.00 000000'),2,73.210000,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10757,'SAVEA',6,stringdate('1995.12.28 00:00.00 000000'),stringdate('1996.01.25 00:00.00 000000'),stringdate('1996.01.15 00:00.00 000000'),1,8.190000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10758,'RICSU',3,stringdate('1995.12.29 00:00.00 000000'),stringdate('1996.01.26 00:00.00 000000'),stringdate('1996.01.04 00:00.00 000000'),3,138.170000,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10759,'ANATR',3,stringdate('1995.12.29 00:00.00 000000'),stringdate('1996.01.26 00:00.00 000000'),stringdate('1996.01.12 00:00.00 000000'),3,11.990000,'Ana Trujillo Emparedados y helados','Avda. de la Constitucin 2222','Mxico D.F.',NULL,'05021','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10760,'MAISD',4,stringdate('1996.01.01 00:00.00 000000'),stringdate('1996.01.29 00:00.00 000000'),stringdate('1996.01.10 00:00.00 000000'),1,155.640000,'Maison Dewey','Rue Joseph-Bens 532','Bruxelles',NULL,'B-1180','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10761,'RATTC',5,stringdate('1996.01.02 00:00.00 000000'),stringdate('1996.01.30 00:00.00 000000'),stringdate('1996.01.08 00:00.00 000000'),2,18.660000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10762,'FOLKO',3,stringdate('1996.01.02 00:00.00 000000'),stringdate('1996.01.30 00:00.00 000000'),stringdate('1996.01.09 00:00.00 000000'),1,328.740000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10763,'FOLIG',3,stringdate('1996.01.03 00:00.00 000000'),stringdate('1996.01.31 00:00.00 000000'),stringdate('1996.01.08 00:00.00 000000'),3,37.350000,'Folies gourmandes','184, chausse de Tournai','Lille',NULL,'59000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10764,'ERNSH',6,stringdate('1996.01.03 00:00.00 000000'),stringdate('1996.01.31 00:00.00 000000'),stringdate('1996.01.08 00:00.00 000000'),3,145.450000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10765,'QUICK',3,stringdate('1996.01.04 00:00.00 000000'),stringdate('1996.02.01 00:00.00 000000'),stringdate('1996.01.09 00:00.00 000000'),3,42.740000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10766,'OTTIK',4,stringdate('1996.01.05 00:00.00 000000'),stringdate('1996.02.02 00:00.00 000000'),stringdate('1996.01.09 00:00.00 000000'),1,157.550000,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10767,'SUPRD',4,stringdate('1996.01.05 00:00.00 000000'),stringdate('1996.02.02 00:00.00 000000'),stringdate('1996.01.15 00:00.00 000000'),3,1.590000,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10768,'AROUT',3,stringdate('1996.01.08 00:00.00 000000'),stringdate('1996.02.05 00:00.00 000000'),stringdate('1996.01.15 00:00.00 000000'),2,146.320000,'Around the Horn','Brook Farm\15\12Stratford St. Mary','Colchester','Essex','CO7 6JX','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10769,'VAFFE',3,stringdate('1996.01.08 00:00.00 000000'),stringdate('1996.02.05 00:00.00 000000'),stringdate('1996.01.12 00:00.00 000000'),1,65.060000,'Vaffeljernet','Smagslget 45','rhus',NULL,'8200','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10770,'HANAR',8,stringdate('1996.01.09 00:00.00 000000'),stringdate('1996.02.06 00:00.00 000000'),stringdate('1996.01.17 00:00.00 000000'),3,5.320000,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10771,'ERNSH',9,stringdate('1996.01.10 00:00.00 000000'),stringdate('1996.02.07 00:00.00 000000'),stringdate('1996.02.02 00:00.00 000000'),2,11.190000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10772,'LEHMS',3,stringdate('1996.01.10 00:00.00 000000'),stringdate('1996.02.07 00:00.00 000000'),stringdate('1996.01.19 00:00.00 000000'),2,91.280000,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M. ',NULL,'60528','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10773,'ERNSH',1,stringdate('1996.01.11 00:00.00 000000'),stringdate('1996.02.08 00:00.00 000000'),stringdate('1996.01.16 00:00.00 000000'),3,96.430000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10774,'FOLKO',4,stringdate('1996.01.11 00:00.00 000000'),stringdate('1996.01.25 00:00.00 000000'),stringdate('1996.01.12 00:00.00 000000'),1,48.200000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10775,'THECR',7,stringdate('1996.01.12 00:00.00 000000'),stringdate('1996.02.09 00:00.00 000000'),stringdate('1996.01.26 00:00.00 000000'),1,20.250000,'The Cracker Box','55 Grizzly Peak Rd.','Butte','MT','59801','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10776,'ERNSH',1,stringdate('1996.01.15 00:00.00 000000'),stringdate('1996.02.12 00:00.00 000000'),stringdate('1996.01.18 00:00.00 000000'),3,351.530000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10777,'GOURL',7,stringdate('1996.01.15 00:00.00 000000'),stringdate('1996.01.29 00:00.00 000000'),stringdate('1996.02.21 00:00.00 000000'),2,3.010000,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10778,'BERGS',3,stringdate('1996.01.16 00:00.00 000000'),stringdate('1996.02.13 00:00.00 000000'),stringdate('1996.01.24 00:00.00 000000'),1,6.790000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10779,'MORGK',3,stringdate('1996.01.16 00:00.00 000000'),stringdate('1996.02.13 00:00.00 000000'),stringdate('1996.02.14 00:00.00 000000'),2,58.130000,'Morgenstern Gesundkost','Heerstr. 22','Leipzig',NULL,'04179','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10780,'LILAS',2,stringdate('1996.01.16 00:00.00 000000'),stringdate('1996.01.30 00:00.00 000000'),stringdate('1996.01.25 00:00.00 000000'),1,42.130000,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10781,'WARTH',2,stringdate('1996.01.17 00:00.00 000000'),stringdate('1996.02.14 00:00.00 000000'),stringdate('1996.01.19 00:00.00 000000'),3,73.160000,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10782,'CACTU',9,stringdate('1996.01.17 00:00.00 000000'),stringdate('1996.02.14 00:00.00 000000'),stringdate('1996.01.22 00:00.00 000000'),3,1.100000,'Cactus Comidas para llevar','Cerrito 333','Buenos Aires',NULL,'1010','Argentina');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10783,'HANAR',4,stringdate('1996.01.18 00:00.00 000000'),stringdate('1996.02.15 00:00.00 000000'),stringdate('1996.01.19 00:00.00 000000'),2,124.980000,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10784,'MAGAA',4,stringdate('1996.01.18 00:00.00 000000'),stringdate('1996.02.15 00:00.00 000000'),stringdate('1996.01.22 00:00.00 000000'),3,70.090000,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10785,'GROSR',1,stringdate('1996.01.18 00:00.00 000000'),stringdate('1996.02.15 00:00.00 000000'),stringdate('1996.01.24 00:00.00 000000'),3,1.510000,'GROSELLA-Restaurante','5 Ave. Los Palos Grandes','Caracas','DF','1081','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10786,'QUEEN',8,stringdate('1996.01.19 00:00.00 000000'),stringdate('1996.02.16 00:00.00 000000'),stringdate('1996.01.23 00:00.00 000000'),1,110.870000,'Queen Cozinha','Alameda dos Canrios, 891','So Paulo','SP','05487-020','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10787,'LAMAI',2,stringdate('1996.01.19 00:00.00 000000'),stringdate('1996.02.02 00:00.00 000000'),stringdate('1996.01.26 00:00.00 000000'),1,249.930000,'La maison d\47Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10788,'QUICK',1,stringdate('1996.01.22 00:00.00 000000'),stringdate('1996.02.19 00:00.00 000000'),stringdate('1996.02.19 00:00.00 000000'),2,42.700000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10789,'FOLIG',1,stringdate('1996.01.22 00:00.00 000000'),stringdate('1996.02.19 00:00.00 000000'),stringdate('1996.01.31 00:00.00 000000'),2,100.600000,'Folies gourmandes','184, chausse de Tournai','Lille',NULL,'59000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10790,'GOURL',6,stringdate('1996.01.22 00:00.00 000000'),stringdate('1996.02.19 00:00.00 000000'),stringdate('1996.01.26 00:00.00 000000'),1,28.230000,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10791,'FRANK',6,stringdate('1996.01.23 00:00.00 000000'),stringdate('1996.02.20 00:00.00 000000'),stringdate('1996.02.01 00:00.00 000000'),2,16.850000,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10792,'WOLZA',1,stringdate('1996.01.23 00:00.00 000000'),stringdate('1996.02.20 00:00.00 000000'),stringdate('1996.01.31 00:00.00 000000'),3,23.790000,'Wolski Zajazd','ul. Filtrowa 68','Warszawa',NULL,'01-012','Poland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10793,'AROUT',3,stringdate('1996.01.24 00:00.00 000000'),stringdate('1996.02.21 00:00.00 000000'),stringdate('1996.02.08 00:00.00 000000'),3,4.520000,'Around the Horn','Brook Farm\15\12Stratford St. Mary','Colchester','Essex','CO7 6JX','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10794,'QUEDE',6,stringdate('1996.01.24 00:00.00 000000'),stringdate('1996.02.21 00:00.00 000000'),stringdate('1996.02.02 00:00.00 000000'),1,21.490000,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10795,'ERNSH',8,stringdate('1996.01.24 00:00.00 000000'),stringdate('1996.02.21 00:00.00 000000'),stringdate('1996.02.20 00:00.00 000000'),2,126.660000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10796,'HILAA',3,stringdate('1996.01.25 00:00.00 000000'),stringdate('1996.02.22 00:00.00 000000'),stringdate('1996.02.14 00:00.00 000000'),1,26.520000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10797,'DRACD',7,stringdate('1996.01.25 00:00.00 000000'),stringdate('1996.02.22 00:00.00 000000'),stringdate('1996.02.05 00:00.00 000000'),2,33.350000,'Drachenblut Delikatessen','Walserweg 21','Aachen',NULL,'52066','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10798,'ISLAT',2,stringdate('1996.01.26 00:00.00 000000'),stringdate('1996.02.23 00:00.00 000000'),stringdate('1996.02.05 00:00.00 000000'),1,2.330000,'Island Trading','Garden House\15\12Crowther Way','Cowes','Isle of Wight','PO31 7PJ','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10799,'KOENE',9,stringdate('1996.01.26 00:00.00 000000'),stringdate('1996.03.08 00:00.00 000000'),stringdate('1996.02.05 00:00.00 000000'),3,30.760000,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10800,'SEVES',1,stringdate('1996.01.26 00:00.00 000000'),stringdate('1996.02.23 00:00.00 000000'),stringdate('1996.02.05 00:00.00 000000'),3,137.440000,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10801,'BOLID',4,stringdate('1996.01.29 00:00.00 000000'),stringdate('1996.02.26 00:00.00 000000'),stringdate('1996.01.31 00:00.00 000000'),2,97.090000,'Blido Comidas preparadas','C/ Araquil, 67','Madrid',NULL,'28023','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10802,'SIMOB',4,stringdate('1996.01.29 00:00.00 000000'),stringdate('1996.02.26 00:00.00 000000'),stringdate('1996.02.02 00:00.00 000000'),2,257.260000,'Simons bistro','Vinbltet 34','Kbenhavn',NULL,'1734','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10803,'WELLI',4,stringdate('1996.01.30 00:00.00 000000'),stringdate('1996.02.27 00:00.00 000000'),stringdate('1996.02.06 00:00.00 000000'),1,55.230000,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10804,'SEVES',6,stringdate('1996.01.30 00:00.00 000000'),stringdate('1996.02.27 00:00.00 000000'),stringdate('1996.02.07 00:00.00 000000'),2,27.330000,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10805,'THEBI',2,stringdate('1996.01.30 00:00.00 000000'),stringdate('1996.02.27 00:00.00 000000'),stringdate('1996.02.09 00:00.00 000000'),3,237.340000,'The Big Cheese','89 Jefferson Way\15\12Suite 2','Portland','OR','97201','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10806,'VICTE',3,stringdate('1996.01.31 00:00.00 000000'),stringdate('1996.02.28 00:00.00 000000'),stringdate('1996.02.05 00:00.00 000000'),2,22.110000,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10807,'FRANS',4,stringdate('1996.01.31 00:00.00 000000'),stringdate('1996.02.28 00:00.00 000000'),stringdate('1996.03.01 00:00.00 000000'),1,1.360000,'Franchi S.p.A.','Via Monte Bianco 34','Torino',NULL,'10100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10808,'OLDWO',2,stringdate('1996.02.01 00:00.00 000000'),stringdate('1996.02.29 00:00.00 000000'),stringdate('1996.02.09 00:00.00 000000'),3,45.530000,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10809,'WELLI',7,stringdate('1996.02.01 00:00.00 000000'),stringdate('1996.02.29 00:00.00 000000'),stringdate('1996.02.07 00:00.00 000000'),1,4.870000,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10810,'LAUGB',2,stringdate('1996.02.01 00:00.00 000000'),stringdate('1996.02.29 00:00.00 000000'),stringdate('1996.02.07 00:00.00 000000'),3,4.330000,'Laughing Bacchus Wine Cellars','2319 Elm St.','Vancouver','BC','V3F 2K1','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10811,'LINOD',8,stringdate('1996.02.02 00:00.00 000000'),stringdate('1996.03.01 00:00.00 000000'),stringdate('1996.02.08 00:00.00 000000'),1,31.220000,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10812,'REGGC',5,stringdate('1996.02.02 00:00.00 000000'),stringdate('1996.03.01 00:00.00 000000'),stringdate('1996.02.12 00:00.00 000000'),1,59.780000,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10813,'RICAR',1,stringdate('1996.02.05 00:00.00 000000'),stringdate('1996.03.04 00:00.00 000000'),stringdate('1996.02.09 00:00.00 000000'),1,47.380000,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10814,'VICTE',3,stringdate('1996.02.05 00:00.00 000000'),stringdate('1996.03.04 00:00.00 000000'),stringdate('1996.02.14 00:00.00 000000'),3,130.940000,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10815,'SAVEA',2,stringdate('1996.02.05 00:00.00 000000'),stringdate('1996.03.04 00:00.00 000000'),stringdate('1996.02.14 00:00.00 000000'),3,14.620000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10816,'GREAL',4,stringdate('1996.02.06 00:00.00 000000'),stringdate('1996.03.05 00:00.00 000000'),stringdate('1996.03.06 00:00.00 000000'),2,719.780000,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10817,'KOENE',3,stringdate('1996.02.06 00:00.00 000000'),stringdate('1996.02.20 00:00.00 000000'),stringdate('1996.02.13 00:00.00 000000'),2,306.070000,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10818,'MAGAA',7,stringdate('1996.02.07 00:00.00 000000'),stringdate('1996.03.06 00:00.00 000000'),stringdate('1996.02.12 00:00.00 000000'),3,65.480000,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10819,'CACTU',2,stringdate('1996.02.07 00:00.00 000000'),stringdate('1996.03.06 00:00.00 000000'),stringdate('1996.02.16 00:00.00 000000'),3,19.760000,'Cactus Comidas para llevar','Cerrito 333','Buenos Aires',NULL,'1010','Argentina');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10820,'RATTC',3,stringdate('1996.02.07 00:00.00 000000'),stringdate('1996.03.06 00:00.00 000000'),stringdate('1996.02.13 00:00.00 000000'),2,37.520000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10821,'SPLIR',1,stringdate('1996.02.08 00:00.00 000000'),stringdate('1996.03.07 00:00.00 000000'),stringdate('1996.02.15 00:00.00 000000'),1,36.680000,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10822,'TRAIH',6,stringdate('1996.02.08 00:00.00 000000'),stringdate('1996.03.07 00:00.00 000000'),stringdate('1996.02.16 00:00.00 000000'),3,7.000000,'Trail\47s Head Gourmet Provisioners','722 DaVinci Blvd.','Kirkland','WA','98034','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10823,'LILAS',5,stringdate('1996.02.09 00:00.00 000000'),stringdate('1996.03.08 00:00.00 000000'),stringdate('1996.02.13 00:00.00 000000'),2,163.970000,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10824,'FOLKO',8,stringdate('1996.02.09 00:00.00 000000'),stringdate('1996.03.08 00:00.00 000000'),stringdate('1996.03.01 00:00.00 000000'),1,1.230000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10825,'DRACD',1,stringdate('1996.02.09 00:00.00 000000'),stringdate('1996.03.08 00:00.00 000000'),stringdate('1996.02.14 00:00.00 000000'),1,79.250000,'Drachenblut Delikatessen','Walserweg 21','Aachen',NULL,'52066','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10826,'BLONP',6,stringdate('1996.02.12 00:00.00 000000'),stringdate('1996.03.11 00:00.00 000000'),stringdate('1996.03.08 00:00.00 000000'),1,7.090000,'Blondel pre et fils','24, place Klber','Strasbourg',NULL,'67000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10827,'BONAP',1,stringdate('1996.02.12 00:00.00 000000'),stringdate('1996.02.26 00:00.00 000000'),stringdate('1996.03.08 00:00.00 000000'),2,63.540000,'Bon app\47','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10828,'RANCH',9,stringdate('1996.02.13 00:00.00 000000'),stringdate('1996.02.27 00:00.00 000000'),stringdate('1996.03.06 00:00.00 000000'),1,90.850000,'Rancho grande','Av. del Libertador 900','Buenos Aires',NULL,'1010','Argentina');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10829,'ISLAT',9,stringdate('1996.02.13 00:00.00 000000'),stringdate('1996.03.12 00:00.00 000000'),stringdate('1996.02.23 00:00.00 000000'),1,154.720000,'Island Trading','Garden House\15\12Crowther Way','Cowes','Isle of Wight','PO31 7PJ','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10830,'TRADH',4,stringdate('1996.02.13 00:00.00 000000'),stringdate('1996.03.26 00:00.00 000000'),stringdate('1996.02.21 00:00.00 000000'),2,81.830000,'Tradio Hipermercados','Av. Ins de Castro, 414','So Paulo','SP','05634-030','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10831,'SANTG',3,stringdate('1996.02.14 00:00.00 000000'),stringdate('1996.03.13 00:00.00 000000'),stringdate('1996.02.23 00:00.00 000000'),2,72.190000,'Sant Gourmet','Erling Skakkes gate 78','Stavern',NULL,'4110','Norway');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10832,'LAMAI',2,stringdate('1996.02.14 00:00.00 000000'),stringdate('1996.03.13 00:00.00 000000'),stringdate('1996.02.19 00:00.00 000000'),2,43.260000,'La maison d\47Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10833,'OTTIK',6,stringdate('1996.02.15 00:00.00 000000'),stringdate('1996.03.14 00:00.00 000000'),stringdate('1996.02.23 00:00.00 000000'),2,71.490000,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10834,'TRADH',1,stringdate('1996.02.15 00:00.00 000000'),stringdate('1996.03.14 00:00.00 000000'),stringdate('1996.02.19 00:00.00 000000'),3,29.780000,'Tradio Hipermercados','Av. Ins de Castro, 414','So Paulo','SP','05634-030','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10835,'ALFKI',1,stringdate('1996.02.15 00:00.00 000000'),stringdate('1996.03.14 00:00.00 000000'),stringdate('1996.02.21 00:00.00 000000'),3,69.530000,'Alfred\47s Futterkiste','Obere Str. 57','Berlin',NULL,'12209','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10836,'ERNSH',7,stringdate('1996.02.16 00:00.00 000000'),stringdate('1996.03.15 00:00.00 000000'),stringdate('1996.02.21 00:00.00 000000'),1,411.880000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10837,'BERGS',9,stringdate('1996.02.16 00:00.00 000000'),stringdate('1996.03.15 00:00.00 000000'),stringdate('1996.02.23 00:00.00 000000'),3,13.320000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10838,'LINOD',3,stringdate('1996.02.19 00:00.00 000000'),stringdate('1996.03.18 00:00.00 000000'),stringdate('1996.02.23 00:00.00 000000'),3,59.280000,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10839,'TRADH',3,stringdate('1996.02.19 00:00.00 000000'),stringdate('1996.03.18 00:00.00 000000'),stringdate('1996.02.22 00:00.00 000000'),3,35.430000,'Tradio Hipermercados','Av. Ins de Castro, 414','So Paulo','SP','05634-030','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10840,'LINOD',4,stringdate('1996.02.19 00:00.00 000000'),stringdate('1996.04.01 00:00.00 000000'),stringdate('1996.03.18 00:00.00 000000'),2,2.710000,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10841,'SUPRD',5,stringdate('1996.02.20 00:00.00 000000'),stringdate('1996.03.19 00:00.00 000000'),stringdate('1996.02.29 00:00.00 000000'),2,424.300000,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10842,'TORTU',1,stringdate('1996.02.20 00:00.00 000000'),stringdate('1996.03.19 00:00.00 000000'),stringdate('1996.02.29 00:00.00 000000'),3,54.420000,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'05033','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10843,'VICTE',4,stringdate('1996.02.21 00:00.00 000000'),stringdate('1996.03.20 00:00.00 000000'),stringdate('1996.02.26 00:00.00 000000'),2,9.260000,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10844,'PICCO',8,stringdate('1996.02.21 00:00.00 000000'),stringdate('1996.03.20 00:00.00 000000'),stringdate('1996.02.26 00:00.00 000000'),2,25.220000,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10845,'QUICK',8,stringdate('1996.02.21 00:00.00 000000'),stringdate('1996.03.06 00:00.00 000000'),stringdate('1996.03.01 00:00.00 000000'),1,212.980000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10846,'SUPRD',2,stringdate('1996.02.22 00:00.00 000000'),stringdate('1996.04.04 00:00.00 000000'),stringdate('1996.02.23 00:00.00 000000'),3,56.460000,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10847,'SAVEA',4,stringdate('1996.02.22 00:00.00 000000'),stringdate('1996.03.07 00:00.00 000000'),stringdate('1996.03.12 00:00.00 000000'),3,487.570000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10848,'CONSH',7,stringdate('1996.02.23 00:00.00 000000'),stringdate('1996.03.22 00:00.00 000000'),stringdate('1996.02.29 00:00.00 000000'),2,38.240000,'Consolidated Holdings','Berkeley Gardens\15\1212  Brewery ','London',NULL,'WX1 6LT','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10849,'KOENE',9,stringdate('1996.02.23 00:00.00 000000'),stringdate('1996.03.22 00:00.00 000000'),stringdate('1996.03.01 00:00.00 000000'),2,0.560000,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10850,'VICTE',1,stringdate('1996.02.23 00:00.00 000000'),stringdate('1996.04.05 00:00.00 000000'),stringdate('1996.03.01 00:00.00 000000'),1,49.190000,'Victuailles en stock','2, rue du Commerce','Lyon',NULL,'69004','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10851,'RICAR',5,stringdate('1996.02.26 00:00.00 000000'),stringdate('1996.03.25 00:00.00 000000'),stringdate('1996.03.04 00:00.00 000000'),1,160.550000,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10852,'RATTC',8,stringdate('1996.02.26 00:00.00 000000'),stringdate('1996.03.11 00:00.00 000000'),stringdate('1996.03.01 00:00.00 000000'),1,174.050000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10853,'BLAUS',9,stringdate('1996.02.27 00:00.00 000000'),stringdate('1996.03.26 00:00.00 000000'),stringdate('1996.03.05 00:00.00 000000'),2,53.830000,'Blauer See Delikatessen','Forsterstr. 57','Mannheim',NULL,'68306','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10854,'ERNSH',3,stringdate('1996.02.27 00:00.00 000000'),stringdate('1996.03.26 00:00.00 000000'),stringdate('1996.03.07 00:00.00 000000'),2,100.220000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10855,'OLDWO',3,stringdate('1996.02.27 00:00.00 000000'),stringdate('1996.03.26 00:00.00 000000'),stringdate('1996.03.06 00:00.00 000000'),1,170.970000,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10856,'ANTON',3,stringdate('1996.02.28 00:00.00 000000'),stringdate('1996.03.27 00:00.00 000000'),stringdate('1996.03.12 00:00.00 000000'),2,58.430000,'Antonio Moreno Taquera','Mataderos  2312','Mxico D.F.',NULL,'05023','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10857,'BERGS',8,stringdate('1996.02.28 00:00.00 000000'),stringdate('1996.03.27 00:00.00 000000'),stringdate('1996.03.08 00:00.00 000000'),2,188.850000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10858,'LACOR',2,stringdate('1996.02.29 00:00.00 000000'),stringdate('1996.03.28 00:00.00 000000'),stringdate('1996.03.05 00:00.00 000000'),1,52.510000,'La corne d\47abondance','67, avenue de l\47Europe','Versailles',NULL,'78000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10859,'FRANK',1,stringdate('1996.02.29 00:00.00 000000'),stringdate('1996.03.28 00:00.00 000000'),stringdate('1996.03.04 00:00.00 000000'),2,76.100000,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10860,'FRANR',3,stringdate('1996.02.29 00:00.00 000000'),stringdate('1996.03.28 00:00.00 000000'),stringdate('1996.03.06 00:00.00 000000'),3,19.260000,'France restauration','54, rue Royale','Nantes',NULL,'44000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10861,'WHITC',4,stringdate('1996.03.01 00:00.00 000000'),stringdate('1996.03.29 00:00.00 000000'),stringdate('1996.03.19 00:00.00 000000'),2,14.930000,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10862,'LEHMS',8,stringdate('1996.03.01 00:00.00 000000'),stringdate('1996.04.12 00:00.00 000000'),stringdate('1996.03.04 00:00.00 000000'),2,53.230000,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M. ',NULL,'60528','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10863,'HILAA',4,stringdate('1996.03.04 00:00.00 000000'),stringdate('1996.04.01 00:00.00 000000'),stringdate('1996.03.19 00:00.00 000000'),2,30.260000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10864,'AROUT',4,stringdate('1996.03.04 00:00.00 000000'),stringdate('1996.04.01 00:00.00 000000'),stringdate('1996.03.11 00:00.00 000000'),2,3.040000,'Around the Horn','Brook Farm\15\12Stratford St. Mary','Colchester','Essex','CO7 6JX','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10865,'QUICK',2,stringdate('1996.03.04 00:00.00 000000'),stringdate('1996.03.18 00:00.00 000000'),stringdate('1996.03.14 00:00.00 000000'),1,348.140000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10866,'BERGS',5,stringdate('1996.03.05 00:00.00 000000'),stringdate('1996.04.02 00:00.00 000000'),stringdate('1996.03.14 00:00.00 000000'),1,109.110000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10867,'LONEP',6,stringdate('1996.03.05 00:00.00 000000'),stringdate('1996.04.16 00:00.00 000000'),stringdate('1996.03.13 00:00.00 000000'),1,1.930000,'Lonesome Pine Restaurant','89 Chiaroscuro Rd.','Portland','OR','97219','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10868,'QUEEN',7,stringdate('1996.03.06 00:00.00 000000'),stringdate('1996.04.03 00:00.00 000000'),stringdate('1996.03.25 00:00.00 000000'),2,191.270000,'Queen Cozinha','Alameda dos Canrios, 891','So Paulo','SP','05487-020','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10869,'SEVES',5,stringdate('1996.03.06 00:00.00 000000'),stringdate('1996.04.03 00:00.00 000000'),stringdate('1996.03.11 00:00.00 000000'),1,143.280000,'Seven Seas Imports','90 Wadhurst Rd.','London',NULL,'OX15 4NB','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10870,'WOLZA',5,stringdate('1996.03.06 00:00.00 000000'),stringdate('1996.04.03 00:00.00 000000'),stringdate('1996.03.15 00:00.00 000000'),3,12.040000,'Wolski Zajazd','ul. Filtrowa 68','Warszawa',NULL,'01-012','Poland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10871,'BONAP',9,stringdate('1996.03.07 00:00.00 000000'),stringdate('1996.04.04 00:00.00 000000'),stringdate('1996.03.12 00:00.00 000000'),2,112.270000,'Bon app\47','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10872,'GODOS',5,stringdate('1996.03.07 00:00.00 000000'),stringdate('1996.04.04 00:00.00 000000'),stringdate('1996.03.11 00:00.00 000000'),2,175.320000,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10873,'WILMK',4,stringdate('1996.03.08 00:00.00 000000'),stringdate('1996.04.05 00:00.00 000000'),stringdate('1996.03.11 00:00.00 000000'),1,0.820000,'Wilman Kala','Keskuskatu 45','Helsinki',NULL,'21240','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10874,'GODOS',5,stringdate('1996.03.08 00:00.00 000000'),stringdate('1996.04.05 00:00.00 000000'),stringdate('1996.03.13 00:00.00 000000'),2,19.580000,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10875,'BERGS',4,stringdate('1996.03.08 00:00.00 000000'),stringdate('1996.04.05 00:00.00 000000'),stringdate('1996.04.02 00:00.00 000000'),2,32.370000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10876,'BONAP',7,stringdate('1996.03.11 00:00.00 000000'),stringdate('1996.04.08 00:00.00 000000'),stringdate('1996.03.14 00:00.00 000000'),3,60.420000,'Bon app\47','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10877,'RICAR',1,stringdate('1996.03.11 00:00.00 000000'),stringdate('1996.04.08 00:00.00 000000'),stringdate('1996.03.21 00:00.00 000000'),1,38.060000,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10878,'QUICK',4,stringdate('1996.03.12 00:00.00 000000'),stringdate('1996.04.09 00:00.00 000000'),stringdate('1996.03.14 00:00.00 000000'),1,46.690000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10879,'WILMK',3,stringdate('1996.03.12 00:00.00 000000'),stringdate('1996.04.09 00:00.00 000000'),stringdate('1996.03.14 00:00.00 000000'),3,8.500000,'Wilman Kala','Keskuskatu 45','Helsinki',NULL,'21240','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10880,'FOLKO',7,stringdate('1996.03.12 00:00.00 000000'),stringdate('1996.04.23 00:00.00 000000'),stringdate('1996.03.20 00:00.00 000000'),1,88.010000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10881,'CACTU',4,stringdate('1996.03.13 00:00.00 000000'),stringdate('1996.04.10 00:00.00 000000'),stringdate('1996.03.20 00:00.00 000000'),1,2.840000,'Cactus Comidas para llevar','Cerrito 333','Buenos Aires',NULL,'1010','Argentina');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10882,'SAVEA',4,stringdate('1996.03.13 00:00.00 000000'),stringdate('1996.04.10 00:00.00 000000'),stringdate('1996.03.22 00:00.00 000000'),3,23.100000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10883,'LONEP',8,stringdate('1996.03.14 00:00.00 000000'),stringdate('1996.04.11 00:00.00 000000'),stringdate('1996.03.22 00:00.00 000000'),3,0.530000,'Lonesome Pine Restaurant','89 Chiaroscuro Rd.','Portland','OR','97219','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10884,'LETSS',4,stringdate('1996.03.14 00:00.00 000000'),stringdate('1996.04.11 00:00.00 000000'),stringdate('1996.03.15 00:00.00 000000'),2,90.970000,'Let\47s Stop N Shop','87 Polk St.\15\12Suite 5','San Francisco','CA','94117','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10885,'SUPRD',6,stringdate('1996.03.14 00:00.00 000000'),stringdate('1996.04.11 00:00.00 000000'),stringdate('1996.03.20 00:00.00 000000'),3,5.640000,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10886,'HANAR',1,stringdate('1996.03.15 00:00.00 000000'),stringdate('1996.04.12 00:00.00 000000'),stringdate('1996.04.01 00:00.00 000000'),1,4.990000,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10887,'GALED',8,stringdate('1996.03.15 00:00.00 000000'),stringdate('1996.04.12 00:00.00 000000'),stringdate('1996.03.18 00:00.00 000000'),3,1.250000,'Galera del gastronmo','Rambla de Catalua, 23','Barcelona',NULL,'8022','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10888,'GODOS',1,stringdate('1996.03.18 00:00.00 000000'),stringdate('1996.04.15 00:00.00 000000'),stringdate('1996.03.25 00:00.00 000000'),2,51.870000,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10889,'RATTC',9,stringdate('1996.03.18 00:00.00 000000'),stringdate('1996.04.15 00:00.00 000000'),stringdate('1996.03.25 00:00.00 000000'),3,280.610000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10890,'DUMON',7,stringdate('1996.03.18 00:00.00 000000'),stringdate('1996.04.15 00:00.00 000000'),stringdate('1996.03.20 00:00.00 000000'),1,32.760000,'Du monde entier','67, rue des Cinquante Otages','Nantes',NULL,'44000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10891,'LEHMS',7,stringdate('1996.03.19 00:00.00 000000'),stringdate('1996.04.16 00:00.00 000000'),stringdate('1996.03.21 00:00.00 000000'),2,20.370000,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M. ',NULL,'60528','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10892,'MAISD',4,stringdate('1996.03.19 00:00.00 000000'),stringdate('1996.04.16 00:00.00 000000'),stringdate('1996.03.21 00:00.00 000000'),2,120.270000,'Maison Dewey','Rue Joseph-Bens 532','Bruxelles',NULL,'B-1180','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10893,'KOENE',9,stringdate('1996.03.20 00:00.00 000000'),stringdate('1996.04.17 00:00.00 000000'),stringdate('1996.03.22 00:00.00 000000'),2,77.780000,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10894,'SAVEA',1,stringdate('1996.03.20 00:00.00 000000'),stringdate('1996.04.17 00:00.00 000000'),stringdate('1996.03.22 00:00.00 000000'),1,116.130000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10895,'ERNSH',3,stringdate('1996.03.20 00:00.00 000000'),stringdate('1996.04.17 00:00.00 000000'),stringdate('1996.03.25 00:00.00 000000'),1,162.750000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10896,'MAISD',7,stringdate('1996.03.21 00:00.00 000000'),stringdate('1996.04.18 00:00.00 000000'),stringdate('1996.03.29 00:00.00 000000'),3,32.450000,'Maison Dewey','Rue Joseph-Bens 532','Bruxelles',NULL,'B-1180','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10897,'HUNGO',3,stringdate('1996.03.21 00:00.00 000000'),stringdate('1996.04.18 00:00.00 000000'),stringdate('1996.03.27 00:00.00 000000'),2,603.540000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10898,'OCEAN',4,stringdate('1996.03.22 00:00.00 000000'),stringdate('1996.04.19 00:00.00 000000'),stringdate('1996.04.05 00:00.00 000000'),2,1.270000,'Ocano Atlntico Ltda.','Ing. Gustavo Moncada 8585\15\12Piso 20-A','Buenos Aires',NULL,'1010','Argentina');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10899,'LILAS',5,stringdate('1996.03.22 00:00.00 000000'),stringdate('1996.04.19 00:00.00 000000'),stringdate('1996.03.28 00:00.00 000000'),3,1.210000,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10900,'WELLI',1,stringdate('1996.03.22 00:00.00 000000'),stringdate('1996.04.19 00:00.00 000000'),stringdate('1996.04.03 00:00.00 000000'),2,1.660000,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10901,'HILAA',4,stringdate('1996.03.25 00:00.00 000000'),stringdate('1996.04.22 00:00.00 000000'),stringdate('1996.03.28 00:00.00 000000'),1,62.090000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10902,'FOLKO',1,stringdate('1996.03.25 00:00.00 000000'),stringdate('1996.04.22 00:00.00 000000'),stringdate('1996.04.02 00:00.00 000000'),1,44.150000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10903,'HANAR',3,stringdate('1996.03.26 00:00.00 000000'),stringdate('1996.04.23 00:00.00 000000'),stringdate('1996.04.03 00:00.00 000000'),3,36.710000,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10904,'WHITC',3,stringdate('1996.03.26 00:00.00 000000'),stringdate('1996.04.23 00:00.00 000000'),stringdate('1996.03.29 00:00.00 000000'),3,162.950000,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10905,'WELLI',9,stringdate('1996.03.26 00:00.00 000000'),stringdate('1996.04.23 00:00.00 000000'),stringdate('1996.04.05 00:00.00 000000'),2,13.720000,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10906,'WOLZA',4,stringdate('1996.03.27 00:00.00 000000'),stringdate('1996.04.10 00:00.00 000000'),stringdate('1996.04.02 00:00.00 000000'),3,26.290000,'Wolski Zajazd','ul. Filtrowa 68','Warszawa',NULL,'01-012','Poland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10907,'SPECD',6,stringdate('1996.03.27 00:00.00 000000'),stringdate('1996.04.24 00:00.00 000000'),stringdate('1996.03.29 00:00.00 000000'),3,9.190000,'Spcialits du monde','25, rue Lauriston','Paris',NULL,'75016','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10908,'REGGC',4,stringdate('1996.03.28 00:00.00 000000'),stringdate('1996.04.25 00:00.00 000000'),stringdate('1996.04.05 00:00.00 000000'),2,32.960000,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10909,'SANTG',1,stringdate('1996.03.28 00:00.00 000000'),stringdate('1996.04.25 00:00.00 000000'),stringdate('1996.04.09 00:00.00 000000'),2,53.050000,'Sant Gourmet','Erling Skakkes gate 78','Stavern',NULL,'4110','Norway');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10910,'WILMK',1,stringdate('1996.03.28 00:00.00 000000'),stringdate('1996.04.25 00:00.00 000000'),stringdate('1996.04.03 00:00.00 000000'),3,38.110000,'Wilman Kala','Keskuskatu 45','Helsinki',NULL,'21240','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10911,'GODOS',3,stringdate('1996.03.28 00:00.00 000000'),stringdate('1996.04.25 00:00.00 000000'),stringdate('1996.04.04 00:00.00 000000'),1,38.190000,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10912,'HUNGO',2,stringdate('1996.03.28 00:00.00 000000'),stringdate('1996.04.25 00:00.00 000000'),stringdate('1996.04.17 00:00.00 000000'),2,580.910000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10913,'QUEEN',4,stringdate('1996.03.28 00:00.00 000000'),stringdate('1996.04.25 00:00.00 000000'),stringdate('1996.04.03 00:00.00 000000'),1,33.050000,'Queen Cozinha','Alameda dos Canrios, 891','So Paulo','SP','05487-020','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10914,'QUEEN',6,stringdate('1996.03.29 00:00.00 000000'),stringdate('1996.04.26 00:00.00 000000'),stringdate('1996.04.01 00:00.00 000000'),1,21.190000,'Queen Cozinha','Alameda dos Canrios, 891','So Paulo','SP','05487-020','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10915,'TORTU',2,stringdate('1996.03.29 00:00.00 000000'),stringdate('1996.04.26 00:00.00 000000'),stringdate('1996.04.01 00:00.00 000000'),2,3.510000,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'05033','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10916,'RANCH',1,stringdate('1996.03.29 00:00.00 000000'),stringdate('1996.04.26 00:00.00 000000'),stringdate('1996.04.08 00:00.00 000000'),2,63.770000,'Rancho grande','Av. del Libertador 900','Buenos Aires',NULL,'1010','Argentina');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10917,'ROMEY',4,stringdate('1996.04.01 00:00.00 000000'),stringdate('1996.04.29 00:00.00 000000'),stringdate('1996.04.10 00:00.00 000000'),2,8.290000,'Romero y tomillo','Gran Va, 1','Madrid',NULL,'28001','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10918,'BOTTM',3,stringdate('1996.04.01 00:00.00 000000'),stringdate('1996.04.29 00:00.00 000000'),stringdate('1996.04.10 00:00.00 000000'),3,48.830000,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10919,'LINOD',2,stringdate('1996.04.01 00:00.00 000000'),stringdate('1996.04.29 00:00.00 000000'),stringdate('1996.04.03 00:00.00 000000'),2,19.800000,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10920,'AROUT',4,stringdate('1996.04.02 00:00.00 000000'),stringdate('1996.04.30 00:00.00 000000'),stringdate('1996.04.08 00:00.00 000000'),2,29.610000,'Around the Horn','Brook Farm\15\12Stratford St. Mary','Colchester','Essex','CO7 6JX','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10921,'VAFFE',1,stringdate('1996.04.02 00:00.00 000000'),stringdate('1996.05.14 00:00.00 000000'),stringdate('1996.04.08 00:00.00 000000'),1,176.480000,'Vaffeljernet','Smagslget 45','rhus',NULL,'8200','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10922,'HANAR',5,stringdate('1996.04.02 00:00.00 000000'),stringdate('1996.04.30 00:00.00 000000'),stringdate('1996.04.04 00:00.00 000000'),3,62.740000,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10923,'LAMAI',7,stringdate('1996.04.02 00:00.00 000000'),stringdate('1996.05.14 00:00.00 000000'),stringdate('1996.04.12 00:00.00 000000'),3,68.260000,'La maison d\47Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10924,'BERGS',3,stringdate('1996.04.03 00:00.00 000000'),stringdate('1996.05.01 00:00.00 000000'),stringdate('1996.05.08 00:00.00 000000'),2,151.520000,'Berglunds snabbkp','Berguvsvgen  8','Lule',NULL,'S-958 22','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10925,'HANAR',3,stringdate('1996.04.03 00:00.00 000000'),stringdate('1996.05.01 00:00.00 000000'),stringdate('1996.04.12 00:00.00 000000'),1,2.270000,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10926,'ANATR',4,stringdate('1996.04.03 00:00.00 000000'),stringdate('1996.05.01 00:00.00 000000'),stringdate('1996.04.10 00:00.00 000000'),3,39.920000,'Ana Trujillo Emparedados y helados','Avda. de la Constitucin 2222','Mxico D.F.',NULL,'05021','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10927,'LACOR',4,stringdate('1996.04.04 00:00.00 000000'),stringdate('1996.05.02 00:00.00 000000'),stringdate('1996.05.08 00:00.00 000000'),1,19.790000,'La corne d\47abondance','67, avenue de l\47Europe','Versailles',NULL,'78000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10928,'GALED',1,stringdate('1996.04.04 00:00.00 000000'),stringdate('1996.05.02 00:00.00 000000'),stringdate('1996.04.17 00:00.00 000000'),1,1.360000,'Galera del gastronmo','Rambla de Catalua, 23','Barcelona',NULL,'8022','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10929,'FRANK',6,stringdate('1996.04.04 00:00.00 000000'),stringdate('1996.05.02 00:00.00 000000'),stringdate('1996.04.11 00:00.00 000000'),1,33.930000,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10930,'SUPRD',4,stringdate('1996.04.05 00:00.00 000000'),stringdate('1996.05.17 00:00.00 000000'),stringdate('1996.04.17 00:00.00 000000'),3,15.550000,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10931,'RICSU',4,stringdate('1996.04.05 00:00.00 000000'),stringdate('1996.04.19 00:00.00 000000'),stringdate('1996.04.18 00:00.00 000000'),2,13.600000,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10932,'BONAP',8,stringdate('1996.04.05 00:00.00 000000'),stringdate('1996.05.03 00:00.00 000000'),stringdate('1996.04.23 00:00.00 000000'),1,134.640000,'Bon app\47','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10933,'ISLAT',6,stringdate('1996.04.05 00:00.00 000000'),stringdate('1996.05.03 00:00.00 000000'),stringdate('1996.04.15 00:00.00 000000'),3,54.150000,'Island Trading','Garden House\15\12Crowther Way','Cowes','Isle of Wight','PO31 7PJ','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10934,'LEHMS',3,stringdate('1996.04.08 00:00.00 000000'),stringdate('1996.05.06 00:00.00 000000'),stringdate('1996.04.11 00:00.00 000000'),3,32.010000,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M. ',NULL,'60528','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10935,'WELLI',4,stringdate('1996.04.08 00:00.00 000000'),stringdate('1996.05.06 00:00.00 000000'),stringdate('1996.04.17 00:00.00 000000'),3,47.590000,'Wellington Importadora','Rua do Mercado, 12','Resende','SP','08737-363','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10936,'GREAL',3,stringdate('1996.04.08 00:00.00 000000'),stringdate('1996.05.06 00:00.00 000000'),stringdate('1996.04.17 00:00.00 000000'),2,33.680000,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10937,'CACTU',7,stringdate('1996.04.09 00:00.00 000000'),stringdate('1996.04.23 00:00.00 000000'),stringdate('1996.04.12 00:00.00 000000'),3,31.510000,'Cactus Comidas para llevar','Cerrito 333','Buenos Aires',NULL,'1010','Argentina');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10938,'QUICK',3,stringdate('1996.04.09 00:00.00 000000'),stringdate('1996.05.07 00:00.00 000000'),stringdate('1996.04.15 00:00.00 000000'),2,31.890000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10939,'MAGAA',2,stringdate('1996.04.09 00:00.00 000000'),stringdate('1996.05.07 00:00.00 000000'),stringdate('1996.04.12 00:00.00 000000'),2,76.330000,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10940,'BONAP',8,stringdate('1996.04.10 00:00.00 000000'),stringdate('1996.05.08 00:00.00 000000'),stringdate('1996.04.22 00:00.00 000000'),3,19.770000,'Bon app\47','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10941,'SAVEA',7,stringdate('1996.04.10 00:00.00 000000'),stringdate('1996.05.08 00:00.00 000000'),stringdate('1996.04.19 00:00.00 000000'),2,400.810000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10942,'REGGC',9,stringdate('1996.04.10 00:00.00 000000'),stringdate('1996.05.08 00:00.00 000000'),stringdate('1996.04.17 00:00.00 000000'),3,17.950000,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10943,'BSBEV',4,stringdate('1996.04.10 00:00.00 000000'),stringdate('1996.05.08 00:00.00 000000'),stringdate('1996.04.18 00:00.00 000000'),2,2.170000,'B\47s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10944,'BOTTM',6,stringdate('1996.04.11 00:00.00 000000'),stringdate('1996.04.25 00:00.00 000000'),stringdate('1996.04.12 00:00.00 000000'),3,52.920000,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10945,'MORGK',4,stringdate('1996.04.11 00:00.00 000000'),stringdate('1996.05.09 00:00.00 000000'),stringdate('1996.04.17 00:00.00 000000'),1,10.220000,'Morgenstern Gesundkost','Heerstr. 22','Leipzig',NULL,'04179','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10946,'VAFFE',1,stringdate('1996.04.11 00:00.00 000000'),stringdate('1996.05.09 00:00.00 000000'),stringdate('1996.04.18 00:00.00 000000'),2,27.200000,'Vaffeljernet','Smagslget 45','rhus',NULL,'8200','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10947,'BSBEV',3,stringdate('1996.04.12 00:00.00 000000'),stringdate('1996.05.10 00:00.00 000000'),stringdate('1996.04.15 00:00.00 000000'),2,3.260000,'B\47s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10948,'GODOS',3,stringdate('1996.04.12 00:00.00 000000'),stringdate('1996.05.10 00:00.00 000000'),stringdate('1996.04.18 00:00.00 000000'),3,23.390000,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10949,'BOTTM',2,stringdate('1996.04.12 00:00.00 000000'),stringdate('1996.05.10 00:00.00 000000'),stringdate('1996.04.16 00:00.00 000000'),3,74.440000,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10950,'MAGAA',1,stringdate('1996.04.15 00:00.00 000000'),stringdate('1996.05.13 00:00.00 000000'),stringdate('1996.04.22 00:00.00 000000'),2,2.500000,'Magazzini Alimentari Riuniti','Via Ludovico il Moro 22','Bergamo',NULL,'24100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10951,'RICSU',9,stringdate('1996.04.15 00:00.00 000000'),stringdate('1996.05.27 00:00.00 000000'),stringdate('1996.05.07 00:00.00 000000'),2,30.850000,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10952,'ALFKI',1,stringdate('1996.04.15 00:00.00 000000'),stringdate('1996.05.27 00:00.00 000000'),stringdate('1996.04.23 00:00.00 000000'),1,40.420000,'Alfred\47s Futterkiste','Obere Str. 57','Berlin',NULL,'12209','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10953,'AROUT',9,stringdate('1996.04.15 00:00.00 000000'),stringdate('1996.04.29 00:00.00 000000'),stringdate('1996.04.24 00:00.00 000000'),2,23.720000,'Around the Horn','Brook Farm\15\12Stratford St. Mary','Colchester','Essex','CO7 6JX','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10954,'LINOD',5,stringdate('1996.04.16 00:00.00 000000'),stringdate('1996.05.28 00:00.00 000000'),stringdate('1996.04.19 00:00.00 000000'),1,27.910000,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10955,'FOLKO',8,stringdate('1996.04.16 00:00.00 000000'),stringdate('1996.05.14 00:00.00 000000'),stringdate('1996.04.19 00:00.00 000000'),2,3.260000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10956,'BLAUS',6,stringdate('1996.04.16 00:00.00 000000'),stringdate('1996.05.28 00:00.00 000000'),stringdate('1996.04.19 00:00.00 000000'),2,44.650000,'Blauer See Delikatessen','Forsterstr. 57','Mannheim',NULL,'68306','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10957,'HILAA',8,stringdate('1996.04.17 00:00.00 000000'),stringdate('1996.05.15 00:00.00 000000'),stringdate('1996.04.26 00:00.00 000000'),3,105.360000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10958,'OCEAN',7,stringdate('1996.04.17 00:00.00 000000'),stringdate('1996.05.15 00:00.00 000000'),stringdate('1996.04.26 00:00.00 000000'),2,49.560000,'Ocano Atlntico Ltda.','Ing. Gustavo Moncada 8585\15\12Piso 20-A','Buenos Aires',NULL,'1010','Argentina');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10959,'GOURL',6,stringdate('1996.04.17 00:00.00 000000'),stringdate('1996.05.29 00:00.00 000000'),stringdate('1996.04.22 00:00.00 000000'),2,4.980000,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10960,'HILAA',3,stringdate('1996.04.18 00:00.00 000000'),stringdate('1996.05.02 00:00.00 000000'),stringdate('1996.05.08 00:00.00 000000'),1,2.080000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10961,'QUEEN',8,stringdate('1996.04.18 00:00.00 000000'),stringdate('1996.05.16 00:00.00 000000'),stringdate('1996.04.29 00:00.00 000000'),1,104.470000,'Queen Cozinha','Alameda dos Canrios, 891','So Paulo','SP','05487-020','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10962,'QUICK',8,stringdate('1996.04.18 00:00.00 000000'),stringdate('1996.05.16 00:00.00 000000'),stringdate('1996.04.22 00:00.00 000000'),2,275.790000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10963,'FURIB',9,stringdate('1996.04.18 00:00.00 000000'),stringdate('1996.05.16 00:00.00 000000'),stringdate('1996.04.25 00:00.00 000000'),3,2.700000,'Furia Bacalhau e Frutos do Mar','Jardim das rosas n. 32','Lisboa',NULL,'1675','Portugal');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10964,'SPECD',3,stringdate('1996.04.19 00:00.00 000000'),stringdate('1996.05.17 00:00.00 000000'),stringdate('1996.04.23 00:00.00 000000'),2,87.380000,'Spcialits du monde','25, rue Lauriston','Paris',NULL,'75016','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10965,'OLDWO',6,stringdate('1996.04.19 00:00.00 000000'),stringdate('1996.05.17 00:00.00 000000'),stringdate('1996.04.29 00:00.00 000000'),3,144.380000,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10966,'CHOPS',4,stringdate('1996.04.19 00:00.00 000000'),stringdate('1996.05.17 00:00.00 000000'),stringdate('1996.05.08 00:00.00 000000'),1,27.190000,'Chop-suey Chinese','Hauptstr. 31','Bern',NULL,'3012','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10967,'TOMSP',2,stringdate('1996.04.22 00:00.00 000000'),stringdate('1996.05.20 00:00.00 000000'),stringdate('1996.05.02 00:00.00 000000'),2,62.220000,'Toms Spezialitten','Luisenstr. 48','Mnster',NULL,'44087','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10968,'ERNSH',1,stringdate('1996.04.22 00:00.00 000000'),stringdate('1996.05.20 00:00.00 000000'),stringdate('1996.05.01 00:00.00 000000'),3,74.600000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10969,'COMMI',1,stringdate('1996.04.22 00:00.00 000000'),stringdate('1996.05.20 00:00.00 000000'),stringdate('1996.04.29 00:00.00 000000'),2,0.210000,'Comrcio Mineiro','Av. dos Lusadas, 23','So Paulo','SP','05432-043','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10970,'BOLID',9,stringdate('1996.04.23 00:00.00 000000'),stringdate('1996.05.07 00:00.00 000000'),stringdate('1996.05.24 00:00.00 000000'),1,16.160000,'Blido Comidas preparadas','C/ Araquil, 67','Madrid',NULL,'28023','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10971,'FRANR',2,stringdate('1996.04.23 00:00.00 000000'),stringdate('1996.05.21 00:00.00 000000'),stringdate('1996.05.02 00:00.00 000000'),2,121.820000,'France restauration','54, rue Royale','Nantes',NULL,'44000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10972,'LACOR',4,stringdate('1996.04.23 00:00.00 000000'),stringdate('1996.05.21 00:00.00 000000'),stringdate('1996.04.25 00:00.00 000000'),2,0.020000,'La corne d\47abondance','67, avenue de l\47Europe','Versailles',NULL,'78000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10973,'LACOR',6,stringdate('1996.04.23 00:00.00 000000'),stringdate('1996.05.21 00:00.00 000000'),stringdate('1996.04.26 00:00.00 000000'),2,15.170000,'La corne d\47abondance','67, avenue de l\47Europe','Versailles',NULL,'78000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10974,'SPLIR',3,stringdate('1996.04.24 00:00.00 000000'),stringdate('1996.05.08 00:00.00 000000'),stringdate('1996.05.03 00:00.00 000000'),3,12.960000,'Split Rail Beer & Ale','P.O. Box 555','Lander','WY','82520','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10975,'BOTTM',1,stringdate('1996.04.24 00:00.00 000000'),stringdate('1996.05.22 00:00.00 000000'),stringdate('1996.04.26 00:00.00 000000'),3,32.270000,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10976,'HILAA',1,stringdate('1996.04.24 00:00.00 000000'),stringdate('1996.06.05 00:00.00 000000'),stringdate('1996.05.03 00:00.00 000000'),1,37.970000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10977,'FOLKO',8,stringdate('1996.04.25 00:00.00 000000'),stringdate('1996.05.23 00:00.00 000000'),stringdate('1996.05.10 00:00.00 000000'),3,208.500000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10978,'MAISD',9,stringdate('1996.04.25 00:00.00 000000'),stringdate('1996.05.23 00:00.00 000000'),stringdate('1996.05.23 00:00.00 000000'),2,32.820000,'Maison Dewey','Rue Joseph-Bens 532','Bruxelles',NULL,'B-1180','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10979,'ERNSH',8,stringdate('1996.04.25 00:00.00 000000'),stringdate('1996.05.23 00:00.00 000000'),stringdate('1996.04.30 00:00.00 000000'),2,353.070000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10980,'FOLKO',4,stringdate('1996.04.26 00:00.00 000000'),stringdate('1996.06.07 00:00.00 000000'),stringdate('1996.05.17 00:00.00 000000'),1,1.260000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10981,'HANAR',1,stringdate('1996.04.26 00:00.00 000000'),stringdate('1996.05.24 00:00.00 000000'),stringdate('1996.05.02 00:00.00 000000'),2,193.370000,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10982,'BOTTM',2,stringdate('1996.04.26 00:00.00 000000'),stringdate('1996.05.24 00:00.00 000000'),stringdate('1996.05.08 00:00.00 000000'),1,14.010000,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10983,'SAVEA',2,stringdate('1996.04.26 00:00.00 000000'),stringdate('1996.05.24 00:00.00 000000'),stringdate('1996.05.06 00:00.00 000000'),2,657.540000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10984,'SAVEA',1,stringdate('1996.04.29 00:00.00 000000'),stringdate('1996.05.27 00:00.00 000000'),stringdate('1996.05.03 00:00.00 000000'),3,211.220000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10985,'HUNGO',2,stringdate('1996.04.29 00:00.00 000000'),stringdate('1996.05.27 00:00.00 000000'),stringdate('1996.05.02 00:00.00 000000'),1,91.510000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10986,'OCEAN',8,stringdate('1996.04.29 00:00.00 000000'),stringdate('1996.05.27 00:00.00 000000'),stringdate('1996.05.21 00:00.00 000000'),2,217.860000,'Ocano Atlntico Ltda.','Ing. Gustavo Moncada 8585\15\12Piso 20-A','Buenos Aires',NULL,'1010','Argentina');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10987,'EASTC',8,stringdate('1996.04.30 00:00.00 000000'),stringdate('1996.05.28 00:00.00 000000'),stringdate('1996.05.06 00:00.00 000000'),1,185.480000,'Eastern Connection','35 King George','London',NULL,'WX3 6FW','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10988,'RATTC',3,stringdate('1996.04.30 00:00.00 000000'),stringdate('1996.05.28 00:00.00 000000'),stringdate('1996.05.10 00:00.00 000000'),2,61.140000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10989,'QUEDE',2,stringdate('1996.04.30 00:00.00 000000'),stringdate('1996.05.28 00:00.00 000000'),stringdate('1996.05.02 00:00.00 000000'),1,34.760000,'Que Delcia','Rua da Panificadora, 12','Rio de Janeiro','RJ','02389-673','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10990,'ERNSH',2,stringdate('1996.05.01 00:00.00 000000'),stringdate('1996.06.12 00:00.00 000000'),stringdate('1996.05.07 00:00.00 000000'),3,117.610000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10991,'QUICK',1,stringdate('1996.05.01 00:00.00 000000'),stringdate('1996.05.29 00:00.00 000000'),stringdate('1996.05.07 00:00.00 000000'),1,38.510000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10992,'THEBI',1,stringdate('1996.05.01 00:00.00 000000'),stringdate('1996.05.29 00:00.00 000000'),stringdate('1996.05.03 00:00.00 000000'),3,4.270000,'The Big Cheese','89 Jefferson Way\15\12Suite 2','Portland','OR','97201','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10993,'FOLKO',7,stringdate('1996.05.01 00:00.00 000000'),stringdate('1996.05.29 00:00.00 000000'),stringdate('1996.05.10 00:00.00 000000'),3,8.810000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10994,'VAFFE',2,stringdate('1996.05.02 00:00.00 000000'),stringdate('1996.05.16 00:00.00 000000'),stringdate('1996.05.09 00:00.00 000000'),3,65.530000,'Vaffeljernet','Smagslget 45','rhus',NULL,'8200','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10995,'PERIC',1,stringdate('1996.05.02 00:00.00 000000'),stringdate('1996.05.30 00:00.00 000000'),stringdate('1996.05.06 00:00.00 000000'),3,46.000000,'Pericles Comidas clsicas','Calle Dr. Jorge Cash 321','Mxico D.F.',NULL,'05033','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10996,'QUICK',4,stringdate('1996.05.02 00:00.00 000000'),stringdate('1996.05.30 00:00.00 000000'),stringdate('1996.05.10 00:00.00 000000'),2,1.120000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10997,'LILAS',8,stringdate('1996.05.03 00:00.00 000000'),stringdate('1996.06.14 00:00.00 000000'),stringdate('1996.05.13 00:00.00 000000'),2,73.910000,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10998,'WOLZA',8,stringdate('1996.05.03 00:00.00 000000'),stringdate('1996.05.17 00:00.00 000000'),stringdate('1996.05.17 00:00.00 000000'),2,20.310000,'Wolski Zajazd','ul. Filtrowa 68','Warszawa',NULL,'01-012','Poland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(10999,'OTTIK',6,stringdate('1996.05.03 00:00.00 000000'),stringdate('1996.05.31 00:00.00 000000'),stringdate('1996.05.10 00:00.00 000000'),2,96.350000,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11000,'RATTC',2,stringdate('1996.05.06 00:00.00 000000'),stringdate('1996.06.03 00:00.00 000000'),stringdate('1996.05.14 00:00.00 000000'),3,55.120000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11001,'FOLKO',2,stringdate('1996.05.06 00:00.00 000000'),stringdate('1996.06.03 00:00.00 000000'),stringdate('1996.05.14 00:00.00 000000'),2,197.300000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11002,'SAVEA',4,stringdate('1996.05.06 00:00.00 000000'),stringdate('1996.06.03 00:00.00 000000'),stringdate('1996.05.16 00:00.00 000000'),1,141.160000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11003,'THECR',3,stringdate('1996.05.06 00:00.00 000000'),stringdate('1996.06.03 00:00.00 000000'),stringdate('1996.05.08 00:00.00 000000'),3,14.910000,'The Cracker Box','55 Grizzly Peak Rd.','Butte','MT','59801','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11004,'MAISD',3,stringdate('1996.05.07 00:00.00 000000'),stringdate('1996.06.04 00:00.00 000000'),stringdate('1996.05.20 00:00.00 000000'),1,44.840000,'Maison Dewey','Rue Joseph-Bens 532','Bruxelles',NULL,'B-1180','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11005,'WILMK',2,stringdate('1996.05.07 00:00.00 000000'),stringdate('1996.06.04 00:00.00 000000'),stringdate('1996.05.10 00:00.00 000000'),1,0.750000,'Wilman Kala','Keskuskatu 45','Helsinki',NULL,'21240','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11006,'GREAL',3,stringdate('1996.05.07 00:00.00 000000'),stringdate('1996.06.04 00:00.00 000000'),stringdate('1996.05.15 00:00.00 000000'),2,25.190000,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11007,'PRINI',8,stringdate('1996.05.08 00:00.00 000000'),stringdate('1996.06.05 00:00.00 000000'),stringdate('1996.05.13 00:00.00 000000'),2,202.240000,'Princesa Isabel Vinhos','Estrada da sade n. 58','Lisboa',NULL,'1756','Portugal');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11008,'ERNSH',7,stringdate('1996.05.08 00:00.00 000000'),stringdate('1996.06.05 00:00.00 000000'),NULL,3,79.460000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11009,'GODOS',2,stringdate('1996.05.08 00:00.00 000000'),stringdate('1996.06.05 00:00.00 000000'),stringdate('1996.05.10 00:00.00 000000'),1,59.110000,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11010,'REGGC',2,stringdate('1996.05.09 00:00.00 000000'),stringdate('1996.06.06 00:00.00 000000'),stringdate('1996.05.21 00:00.00 000000'),2,28.710000,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11011,'ALFKI',3,stringdate('1996.05.09 00:00.00 000000'),stringdate('1996.06.06 00:00.00 000000'),stringdate('1996.05.13 00:00.00 000000'),1,1.210000,'Alfred\47s Futterkiste','Obere Str. 57','Berlin',NULL,'12209','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11012,'FRANK',1,stringdate('1996.05.09 00:00.00 000000'),stringdate('1996.05.23 00:00.00 000000'),stringdate('1996.05.17 00:00.00 000000'),3,242.950000,'Frankenversand','Berliner Platz 43','Mnchen',NULL,'80805','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11013,'ROMEY',2,stringdate('1996.05.09 00:00.00 000000'),stringdate('1996.06.06 00:00.00 000000'),stringdate('1996.05.10 00:00.00 000000'),1,32.990000,'Romero y tomillo','Gran Va, 1','Madrid',NULL,'28001','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11014,'LINOD',2,stringdate('1996.05.10 00:00.00 000000'),stringdate('1996.06.07 00:00.00 000000'),stringdate('1996.05.15 00:00.00 000000'),3,23.600000,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11015,'SANTG',2,stringdate('1996.05.10 00:00.00 000000'),stringdate('1996.05.24 00:00.00 000000'),stringdate('1996.05.20 00:00.00 000000'),2,4.620000,'Sant Gourmet','Erling Skakkes gate 78','Stavern',NULL,'4110','Norway');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11016,'AROUT',9,stringdate('1996.05.10 00:00.00 000000'),stringdate('1996.06.07 00:00.00 000000'),stringdate('1996.05.13 00:00.00 000000'),2,33.800000,'Around the Horn','Brook Farm\15\12Stratford St. Mary','Colchester','Essex','CO7 6JX','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11017,'ERNSH',9,stringdate('1996.05.13 00:00.00 000000'),stringdate('1996.06.10 00:00.00 000000'),stringdate('1996.05.20 00:00.00 000000'),2,754.260000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11018,'LONEP',4,stringdate('1996.05.13 00:00.00 000000'),stringdate('1996.06.10 00:00.00 000000'),stringdate('1996.05.16 00:00.00 000000'),2,11.650000,'Lonesome Pine Restaurant','89 Chiaroscuro Rd.','Portland','OR','97219','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11019,'RANCH',6,stringdate('1996.05.13 00:00.00 000000'),stringdate('1996.06.10 00:00.00 000000'),NULL,3,3.170000,'Rancho grande','Av. del Libertador 900','Buenos Aires',NULL,'1010','Argentina');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11020,'OTTIK',2,stringdate('1996.05.14 00:00.00 000000'),stringdate('1996.06.11 00:00.00 000000'),stringdate('1996.05.16 00:00.00 000000'),2,43.300000,'Ottilies Kseladen','Mehrheimerstr. 369','Kln',NULL,'50739','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11021,'QUICK',3,stringdate('1996.05.14 00:00.00 000000'),stringdate('1996.06.11 00:00.00 000000'),stringdate('1996.05.21 00:00.00 000000'),1,297.180000,'QUICK-Stop','Taucherstrae 10','Cunewalde',NULL,'01307','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11022,'HANAR',9,stringdate('1996.05.14 00:00.00 000000'),stringdate('1996.06.11 00:00.00 000000'),stringdate('1996.06.03 00:00.00 000000'),2,6.270000,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11023,'BSBEV',1,stringdate('1996.05.14 00:00.00 000000'),stringdate('1996.05.28 00:00.00 000000'),stringdate('1996.05.24 00:00.00 000000'),2,123.830000,'B\47s Beverages','Fauntleroy Circus','London',NULL,'EC2 5NT','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11024,'EASTC',4,stringdate('1996.05.15 00:00.00 000000'),stringdate('1996.06.12 00:00.00 000000'),stringdate('1996.05.20 00:00.00 000000'),1,74.360000,'Eastern Connection','35 King George','London',NULL,'WX3 6FW','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11025,'WARTH',6,stringdate('1996.05.15 00:00.00 000000'),stringdate('1996.06.12 00:00.00 000000'),stringdate('1996.05.24 00:00.00 000000'),3,29.170000,'Wartian Herkku','Torikatu 38','Oulu',NULL,'90110','Finland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11026,'FRANS',4,stringdate('1996.05.15 00:00.00 000000'),stringdate('1996.06.12 00:00.00 000000'),stringdate('1996.05.28 00:00.00 000000'),1,47.090000,'Franchi S.p.A.','Via Monte Bianco 34','Torino',NULL,'10100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11027,'BOTTM',1,stringdate('1996.05.16 00:00.00 000000'),stringdate('1996.06.13 00:00.00 000000'),stringdate('1996.05.20 00:00.00 000000'),1,52.520000,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11028,'KOENE',2,stringdate('1996.05.16 00:00.00 000000'),stringdate('1996.06.13 00:00.00 000000'),stringdate('1996.05.22 00:00.00 000000'),1,29.590000,'Kniglich Essen','Maubelstr. 90','Brandenburg',NULL,'14776','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11029,'CHOPS',4,stringdate('1996.05.16 00:00.00 000000'),stringdate('1996.06.13 00:00.00 000000'),stringdate('1996.05.27 00:00.00 000000'),1,47.840000,'Chop-suey Chinese','Hauptstr. 31','Bern',NULL,'3012','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11030,'SAVEA',7,stringdate('1996.05.17 00:00.00 000000'),stringdate('1996.06.14 00:00.00 000000'),stringdate('1996.05.27 00:00.00 000000'),2,830.750000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11031,'SAVEA',6,stringdate('1996.05.17 00:00.00 000000'),stringdate('1996.06.14 00:00.00 000000'),stringdate('1996.05.24 00:00.00 000000'),2,227.220000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11032,'WHITC',2,stringdate('1996.05.17 00:00.00 000000'),stringdate('1996.06.14 00:00.00 000000'),stringdate('1996.05.23 00:00.00 000000'),3,606.190000,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11033,'RICSU',7,stringdate('1996.05.17 00:00.00 000000'),stringdate('1996.06.14 00:00.00 000000'),stringdate('1996.05.23 00:00.00 000000'),3,84.740000,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11034,'OLDWO',8,stringdate('1996.05.20 00:00.00 000000'),stringdate('1996.07.01 00:00.00 000000'),stringdate('1996.05.27 00:00.00 000000'),1,40.320000,'Old World Delicatessen','2743 Bering St.','Anchorage','AK','99508','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11035,'SUPRD',2,stringdate('1996.05.20 00:00.00 000000'),stringdate('1996.06.17 00:00.00 000000'),stringdate('1996.05.24 00:00.00 000000'),2,0.170000,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11036,'DRACD',8,stringdate('1996.05.20 00:00.00 000000'),stringdate('1996.06.17 00:00.00 000000'),stringdate('1996.05.22 00:00.00 000000'),3,149.470000,'Drachenblut Delikatessen','Walserweg 21','Aachen',NULL,'52066','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11037,'GODOS',7,stringdate('1996.05.21 00:00.00 000000'),stringdate('1996.06.18 00:00.00 000000'),stringdate('1996.05.27 00:00.00 000000'),1,3.200000,'Godos Cocina Tpica','C/ Romero, 33','Sevilla',NULL,'41101','Spain');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11038,'SUPRD',1,stringdate('1996.05.21 00:00.00 000000'),stringdate('1996.06.18 00:00.00 000000'),stringdate('1996.05.30 00:00.00 000000'),2,29.590000,'Suprmes dlices','Boulevard Tirou, 255','Charleroi',NULL,'B-6000','Belgium');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11039,'LINOD',1,stringdate('1996.05.21 00:00.00 000000'),stringdate('1996.06.18 00:00.00 000000'),NULL,2,65.000000,'LINO-Delicateses','Ave. 5 de Mayo Porlamar','I. de Margarita','Nueva Esparta','4980','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11040,'GREAL',4,stringdate('1996.05.22 00:00.00 000000'),stringdate('1996.06.19 00:00.00 000000'),NULL,3,18.840000,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11041,'CHOPS',3,stringdate('1996.05.22 00:00.00 000000'),stringdate('1996.06.19 00:00.00 000000'),stringdate('1996.05.28 00:00.00 000000'),2,48.220000,'Chop-suey Chinese','Hauptstr. 31','Bern',NULL,'3012','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11042,'COMMI',2,stringdate('1996.05.22 00:00.00 000000'),stringdate('1996.06.05 00:00.00 000000'),stringdate('1996.05.31 00:00.00 000000'),1,29.990000,'Comrcio Mineiro','Av. dos Lusadas, 23','So Paulo','SP','05432-043','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11043,'SPECD',5,stringdate('1996.05.22 00:00.00 000000'),stringdate('1996.06.19 00:00.00 000000'),stringdate('1996.05.29 00:00.00 000000'),2,8.800000,'Spcialits du monde','25, rue Lauriston','Paris',NULL,'75016','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11044,'WOLZA',4,stringdate('1996.05.23 00:00.00 000000'),stringdate('1996.06.20 00:00.00 000000'),stringdate('1996.05.31 00:00.00 000000'),1,8.720000,'Wolski Zajazd','ul. Filtrowa 68','Warszawa',NULL,'01-012','Poland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11045,'BOTTM',6,stringdate('1996.05.23 00:00.00 000000'),stringdate('1996.06.20 00:00.00 000000'),NULL,2,70.580000,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11046,'WANDK',8,stringdate('1996.05.23 00:00.00 000000'),stringdate('1996.06.20 00:00.00 000000'),stringdate('1996.05.24 00:00.00 000000'),2,71.640000,'Die Wandernde Kuh','Adenauerallee 900','Stuttgart',NULL,'70563','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11047,'EASTC',7,stringdate('1996.05.24 00:00.00 000000'),stringdate('1996.06.21 00:00.00 000000'),stringdate('1996.05.31 00:00.00 000000'),3,46.620000,'Eastern Connection','35 King George','London',NULL,'WX3 6FW','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11048,'BOTTM',7,stringdate('1996.05.24 00:00.00 000000'),stringdate('1996.06.21 00:00.00 000000'),stringdate('1996.05.30 00:00.00 000000'),3,24.120000,'Bottom-Dollar Markets','23 Tsawassen Blvd.','Tsawassen','BC','T2F 8M4','Canada');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11049,'GOURL',3,stringdate('1996.05.24 00:00.00 000000'),stringdate('1996.06.21 00:00.00 000000'),stringdate('1996.06.03 00:00.00 000000'),1,8.340000,'Gourmet Lanchonetes','Av. Brasil, 442','Campinas','SP','04876-786','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11050,'FOLKO',8,stringdate('1996.05.27 00:00.00 000000'),stringdate('1996.06.24 00:00.00 000000'),stringdate('1996.06.04 00:00.00 000000'),2,59.410000,'Folk och f HB','kergatan 24','Brcke',NULL,'S-844 67','Sweden');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11051,'LAMAI',7,stringdate('1996.05.27 00:00.00 000000'),stringdate('1996.06.24 00:00.00 000000'),NULL,3,2.790000,'La maison d\47Asie','1 rue Alsace-Lorraine','Toulouse',NULL,'31000','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11052,'HANAR',3,stringdate('1996.05.27 00:00.00 000000'),stringdate('1996.06.24 00:00.00 000000'),stringdate('1996.05.31 00:00.00 000000'),1,67.260000,'Hanari Carnes','Rua do Pao, 67','Rio de Janeiro','RJ','05454-876','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11053,'PICCO',2,stringdate('1996.05.27 00:00.00 000000'),stringdate('1996.06.24 00:00.00 000000'),stringdate('1996.05.29 00:00.00 000000'),2,53.050000,'Piccolo und mehr','Geislweg 14','Salzburg',NULL,'5020','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11054,'CACTU',8,stringdate('1996.05.28 00:00.00 000000'),stringdate('1996.06.25 00:00.00 000000'),NULL,1,0.330000,'Cactus Comidas para llevar','Cerrito 333','Buenos Aires',NULL,'1010','Argentina');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11055,'HILAA',7,stringdate('1996.05.28 00:00.00 000000'),stringdate('1996.06.25 00:00.00 000000'),stringdate('1996.06.04 00:00.00 000000'),2,120.920000,'HILARIN-Abastos','Carrera 22 con Ave. Carlos Soublette #8-35','San Cristbal','Tchira','5022','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11056,'EASTC',8,stringdate('1996.05.28 00:00.00 000000'),stringdate('1996.06.11 00:00.00 000000'),stringdate('1996.05.31 00:00.00 000000'),2,278.960000,'Eastern Connection','35 King George','London',NULL,'WX3 6FW','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11057,'NORTS',3,stringdate('1996.05.29 00:00.00 000000'),stringdate('1996.06.26 00:00.00 000000'),stringdate('1996.05.31 00:00.00 000000'),3,4.130000,'North/South','South House\15\12300 Queensbridge','London',NULL,'SW7 1RZ','United Kingdom');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11058,'BLAUS',9,stringdate('1996.05.29 00:00.00 000000'),stringdate('1996.06.26 00:00.00 000000'),NULL,3,31.140000,'Blauer See Delikatessen','Forsterstr. 57','Mannheim',NULL,'68306','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11059,'RICAR',2,stringdate('1996.05.29 00:00.00 000000'),stringdate('1996.07.10 00:00.00 000000'),NULL,2,85.800000,'Ricardo Adocicados','Av. Copacabana, 267','Rio de Janeiro','RJ','02389-890','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11060,'FRANS',2,stringdate('1996.05.30 00:00.00 000000'),stringdate('1996.06.27 00:00.00 000000'),stringdate('1996.06.03 00:00.00 000000'),2,10.980000,'Franchi S.p.A.','Via Monte Bianco 34','Torino',NULL,'10100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11061,'GREAL',4,stringdate('1996.05.30 00:00.00 000000'),stringdate('1996.07.11 00:00.00 000000'),NULL,3,14.010000,'Great Lakes Food Market','2732 Baker Blvd.','Eugene','OR','97403','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11062,'REGGC',4,stringdate('1996.05.30 00:00.00 000000'),stringdate('1996.06.27 00:00.00 000000'),NULL,2,29.930000,'Reggiani Caseifici','Strada Provinciale 124','Reggio Emilia',NULL,'42100','Italy');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11063,'HUNGO',3,stringdate('1996.05.30 00:00.00 000000'),stringdate('1996.06.27 00:00.00 000000'),stringdate('1996.06.05 00:00.00 000000'),2,81.730000,'Hungry Owl All-Night Grocers','8 Johnstown Road','Cork','Co. Cork',NULL,'Ireland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11064,'SAVEA',1,stringdate('1996.05.31 00:00.00 000000'),stringdate('1996.06.28 00:00.00 000000'),stringdate('1996.06.03 00:00.00 000000'),1,30.090000,'Save-a-lot Markets','187 Suffolk Ln.','Boise','ID','83720','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11065,'LILAS',8,stringdate('1996.05.31 00:00.00 000000'),stringdate('1996.06.28 00:00.00 000000'),NULL,1,12.910000,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11066,'WHITC',7,stringdate('1996.05.31 00:00.00 000000'),stringdate('1996.06.28 00:00.00 000000'),stringdate('1996.06.03 00:00.00 000000'),2,44.720000,'White Clover Markets','1029 - 12th Ave. S.','Seattle','WA','98124','United States');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11067,'DRACD',1,stringdate('1996.06.03 00:00.00 000000'),stringdate('1996.06.17 00:00.00 000000'),stringdate('1996.06.05 00:00.00 000000'),2,7.980000,'Drachenblut Delikatessen','Walserweg 21','Aachen',NULL,'52066','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11068,'QUEEN',8,stringdate('1996.06.03 00:00.00 000000'),stringdate('1996.07.01 00:00.00 000000'),NULL,2,81.750000,'Queen Cozinha','Alameda dos Canrios, 891','So Paulo','SP','05487-020','Brazil');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11069,'TORTU',1,stringdate('1996.06.03 00:00.00 000000'),stringdate('1996.07.01 00:00.00 000000'),stringdate('1996.06.05 00:00.00 000000'),2,15.670000,'Tortuga Restaurante','Avda. Azteca 123','Mxico D.F.',NULL,'05033','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11070,'LEHMS',2,stringdate('1996.06.04 00:00.00 000000'),stringdate('1996.07.02 00:00.00 000000'),NULL,1,136.000000,'Lehmanns Marktstand','Magazinweg 7','Frankfurt a.M. ',NULL,'60528','Germany');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11071,'LILAS',1,stringdate('1996.06.04 00:00.00 000000'),stringdate('1996.07.02 00:00.00 000000'),NULL,1,0.930000,'LILA-Supermercado','Carrera 52 con Ave. Bolvar #65-98 Llano Largo','Barquisimeto','Lara','3508','Venezuela');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11072,'ERNSH',4,stringdate('1996.06.04 00:00.00 000000'),stringdate('1996.07.02 00:00.00 000000'),NULL,2,258.640000,'Ernst Handel','Kirchgasse 6','Graz',NULL,'8010','Austria');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11073,'PERIC',2,stringdate('1996.06.04 00:00.00 000000'),stringdate('1996.07.02 00:00.00 000000'),NULL,2,24.950000,'Pericles Comidas clsicas','Calle Dr. Jorge Cash 321','Mxico D.F.',NULL,'05033','Mexico');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11074,'SIMOB',7,stringdate('1996.06.05 00:00.00 000000'),stringdate('1996.07.03 00:00.00 000000'),NULL,2,18.440000,'Simons bistro','Vinbltet 34','Kbenhavn',NULL,'1734','Denmark');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11075,'RICSU',8,stringdate('1996.06.05 00:00.00 000000'),stringdate('1996.07.03 00:00.00 000000'),NULL,2,6.190000,'Richter Supermarkt','Starenweg 5','Genve',NULL,'1204','Switzerland');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11076,'BONAP',4,stringdate('1996.06.05 00:00.00 000000'),stringdate('1996.07.03 00:00.00 000000'),NULL,2,38.280000,'Bon app\47','12, rue des Bouchers','Marseille',NULL,'13008','France');
INSERT soft Orders(OrderID,CustomerID,EmployeeID,OrderDate,RequiredDate,ShippedDate,ShipVia,Freight,ShipName,ShipAddress,ShipCity,ShipRegion,ShipPostalCode,ShipCountry) VALUES(11077,'RATTC',1,stringdate('1996.06.05 00:00.00 000000'),stringdate('1996.07.03 00:00.00 000000'),NULL,2,8.530000,'Rattlesnake Canyon Grocery','2817 Milton Dr.','Albuquerque','NM','87110','United States');

select count(*) from Orders;
--ECHO BOTH $IF $EQU $LAST[1] 830 "PASSED" "***FAILED";
--SET ARGV[$LIF] $+ $ARGV[$LIF] 1;
--ECHO BOTH ": " $LAST[1] " Orders loaded\n";

--
--
--

DB.DBA.exec_no_error('DELETE FROM Order_Details');

INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10248,11,14.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10248,42,9.800000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10248,72,34.800000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10249,14,18.600000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10249,51,42.400000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10250,41,7.700000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10250,51,42.400000,35,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10250,65,16.800000,15,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10251,22,16.800000,6,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10251,57,15.600000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10251,65,16.800000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10252,20,64.800000,40,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10252,33,2.000000,25,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10252,60,27.200000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10253,31,10.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10253,39,14.400000,42,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10253,49,16.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10254,24,3.600000,15,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10254,55,19.200000,21,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10254,74,8.000000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10255,2,15.200000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10255,16,13.900000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10255,36,15.200000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10255,59,44.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10256,53,26.200000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10256,77,10.400000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10257,27,35.100000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10257,39,14.400000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10257,77,10.400000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10258,2,15.200000,50,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10258,5,17.000000,65,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10258,32,25.600000,6,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10259,21,8.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10259,37,20.800000,1,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10260,41,7.700000,16,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10260,57,15.600000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10260,62,39.400000,15,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10260,70,12.000000,21,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10261,21,8.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10261,35,14.400000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10262,5,17.000000,12,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10262,7,24.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10262,56,30.400000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10263,16,13.900000,60,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10263,24,3.600000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10263,30,20.700000,60,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10263,74,8.000000,36,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10264,2,15.200000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10264,41,7.700000,25,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10265,17,31.200000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10265,70,12.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10266,12,30.400000,12,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10267,40,14.700000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10267,59,44.000000,70,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10267,76,14.400000,15,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10268,29,99.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10268,72,27.800000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10269,33,2.000000,60,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10269,72,27.800000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10270,36,15.200000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10270,43,36.800000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10271,33,2.000000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10272,20,64.800000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10272,31,10.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10272,72,27.800000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10273,10,24.800000,24,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10273,31,10.000000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10273,33,2.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10273,40,14.700000,60,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10273,76,14.400000,33,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10274,71,17.200000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10274,72,27.800000,7,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10275,24,3.600000,12,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10275,59,44.000000,6,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10276,10,24.800000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10276,13,4.800000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10277,28,36.400000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10277,62,39.400000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10278,44,15.500000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10278,59,44.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10278,63,35.100000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10278,73,12.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10279,17,31.200000,15,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10280,24,3.600000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10280,55,19.200000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10280,75,6.200000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10281,19,7.300000,1,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10281,24,3.600000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10281,35,14.400000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10282,30,20.700000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10282,57,15.600000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10283,15,12.400000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10283,19,7.300000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10283,60,27.200000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10283,72,27.800000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10284,27,35.100000,15,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10284,44,15.500000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10284,60,27.200000,20,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10284,67,11.200000,5,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10285,1,14.400000,45,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10285,40,14.700000,40,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10285,53,26.200000,36,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10286,35,14.400000,100,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10286,62,39.400000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10287,16,13.900000,40,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10287,34,11.200000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10287,46,9.600000,15,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10288,54,5.900000,10,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10288,68,10.000000,3,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10289,3,8.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10289,64,26.600000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10290,5,17.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10290,29,99.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10290,49,16.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10290,77,10.400000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10291,13,4.800000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10291,44,15.500000,24,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10291,51,42.400000,2,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10292,20,64.800000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10293,18,50.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10293,24,3.600000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10293,63,35.100000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10293,75,6.200000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10294,1,14.400000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10294,17,31.200000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10294,43,36.800000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10294,60,27.200000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10294,75,6.200000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10295,56,30.400000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10296,11,16.800000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10296,16,13.900000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10296,69,28.800000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10297,39,14.400000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10297,72,27.800000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10298,2,15.200000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10298,36,15.200000,40,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10298,59,44.000000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10298,62,39.400000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10299,19,7.300000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10299,70,12.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10300,66,13.600000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10300,68,10.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10301,40,14.700000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10301,56,30.400000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10302,17,31.200000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10302,28,36.400000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10302,43,36.800000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10303,40,14.700000,40,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10303,65,16.800000,30,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10303,68,10.000000,15,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10304,49,16.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10304,59,44.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10304,71,17.200000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10305,18,50.000000,25,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10305,29,99.000000,25,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10305,39,14.400000,30,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10306,30,20.700000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10306,53,26.200000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10306,54,5.900000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10307,62,39.400000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10307,68,10.000000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10308,69,28.800000,1,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10308,70,12.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10309,4,17.600000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10309,6,20.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10309,42,11.200000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10309,43,36.800000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10309,71,17.200000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10310,16,13.900000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10310,62,39.400000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10311,42,11.200000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10311,69,28.800000,7,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10312,28,36.400000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10312,43,36.800000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10312,53,26.200000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10312,75,6.200000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10313,36,15.200000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10314,32,25.600000,40,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10314,58,10.600000,30,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10314,62,39.400000,25,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10315,34,11.200000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10315,70,12.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10316,41,7.700000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10316,62,39.400000,70,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10317,1,14.400000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10318,41,7.700000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10318,76,14.400000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10319,17,31.200000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10319,28,36.400000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10319,76,14.400000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10320,71,17.200000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10321,35,14.400000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10322,52,5.600000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10323,15,12.400000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10323,25,11.200000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10323,39,14.400000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10324,16,13.900000,21,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10324,35,14.400000,70,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10324,46,9.600000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10324,59,44.000000,40,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10324,63,35.100000,80,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10325,6,20.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10325,13,4.800000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10325,14,18.600000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10325,31,10.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10325,72,27.800000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10326,4,17.600000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10326,57,15.600000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10326,75,6.200000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10327,2,15.200000,25,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10327,11,16.800000,50,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10327,30,20.700000,35,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10327,58,10.600000,30,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10328,59,44.000000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10328,65,16.800000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10328,68,10.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10329,19,7.300000,10,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10329,30,20.700000,8,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10329,38,210.800000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10329,56,30.400000,12,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10330,26,24.900000,50,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10330,72,27.800000,25,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10331,54,5.900000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10332,18,50.000000,40,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10332,42,11.200000,10,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10332,47,7.600000,16,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10333,14,18.600000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10333,21,8.000000,10,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10333,71,17.200000,40,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10334,52,5.600000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10334,68,10.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10335,2,15.200000,7,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10335,31,10.000000,25,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10335,32,25.600000,6,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10335,51,42.400000,48,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10336,4,17.600000,18,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10337,23,7.200000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10337,26,24.900000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10337,36,15.200000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10337,37,20.800000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10337,72,27.800000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10338,17,31.200000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10338,30,20.700000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10339,4,17.600000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10339,17,31.200000,70,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10339,62,39.400000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10340,18,50.000000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10340,41,7.700000,12,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10340,43,36.800000,40,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10341,33,2.000000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10341,59,44.000000,9,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10342,2,15.200000,24,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10342,31,10.000000,56,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10342,36,15.200000,40,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10342,55,19.200000,40,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10343,64,26.600000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10343,68,10.000000,4,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10343,76,14.400000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10344,4,17.600000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10344,8,32.000000,70,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10345,8,32.000000,70,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10345,19,7.300000,80,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10345,42,11.200000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10346,17,31.200000,36,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10346,56,30.400000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10347,25,11.200000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10347,39,14.400000,50,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10347,40,14.700000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10347,75,6.200000,6,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10348,1,14.400000,15,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10348,23,7.200000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10349,54,5.900000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10350,50,13.000000,15,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10350,69,28.800000,18,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10351,38,210.800000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10351,41,7.700000,13,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10351,44,15.500000,77,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10351,65,16.800000,10,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10352,24,3.600000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10352,54,5.900000,20,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10353,11,16.800000,12,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10353,38,210.800000,50,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10354,1,14.400000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10354,29,99.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10355,24,3.600000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10355,57,15.600000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10356,31,10.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10356,55,19.200000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10356,69,28.800000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10357,10,24.800000,30,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10357,26,24.900000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10357,60,27.200000,8,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10358,24,3.600000,10,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10358,34,11.200000,10,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10358,36,15.200000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10359,16,13.900000,56,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10359,31,10.000000,70,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10359,60,27.200000,80,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10360,28,36.400000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10360,29,99.000000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10360,38,210.800000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10360,49,16.000000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10360,54,5.900000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10361,39,14.400000,54,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10361,60,27.200000,55,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10362,25,11.200000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10362,51,42.400000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10362,54,5.900000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10363,31,10.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10363,75,6.200000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10363,76,14.400000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10364,69,28.800000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10364,71,17.200000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10365,11,16.800000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10366,65,16.800000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10366,77,10.400000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10367,34,11.200000,36,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10367,54,5.900000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10367,65,16.800000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10367,77,10.400000,7,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10368,21,8.000000,5,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10368,28,36.400000,13,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10368,57,15.600000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10368,64,26.600000,35,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10369,29,99.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10369,56,30.400000,18,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10370,1,14.400000,15,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10370,64,26.600000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10370,74,8.000000,20,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10371,36,15.200000,6,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10372,20,64.800000,12,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10372,38,210.800000,40,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10372,60,27.200000,70,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10372,72,27.800000,42,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10373,58,10.600000,80,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10373,71,17.200000,50,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10374,31,10.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10374,58,10.600000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10375,14,18.600000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10375,54,5.900000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10376,31,10.000000,42,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10377,28,36.400000,20,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10377,39,14.400000,20,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10378,71,17.200000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10379,41,7.700000,8,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10379,63,35.100000,16,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10379,65,16.800000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10380,30,20.700000,18,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10380,53,26.200000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10380,60,27.200000,6,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10380,70,12.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10381,74,8.000000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10382,5,17.000000,32,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10382,18,50.000000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10382,29,99.000000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10382,33,2.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10382,74,8.000000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10383,13,4.800000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10383,50,13.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10383,56,30.400000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10384,20,64.800000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10384,60,27.200000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10385,7,24.000000,10,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10385,60,27.200000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10385,68,10.000000,8,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10386,24,3.600000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10386,34,11.200000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10387,24,3.600000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10387,28,36.400000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10387,59,44.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10387,71,17.200000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10388,45,7.600000,15,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10388,52,5.600000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10388,53,26.200000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10389,10,24.800000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10389,55,19.200000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10389,62,39.400000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10389,70,12.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10390,31,10.000000,60,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10390,35,14.400000,40,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10390,46,9.600000,45,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10390,72,27.800000,24,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10391,13,4.800000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10392,69,28.800000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10393,2,15.200000,25,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10393,14,18.600000,42,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10393,25,11.200000,7,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10393,26,24.900000,70,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10393,31,10.000000,32,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10394,13,4.800000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10394,62,39.400000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10395,46,9.600000,28,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10395,53,26.200000,70,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10395,69,28.800000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10396,23,7.200000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10396,71,17.200000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10396,72,27.800000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10397,21,8.000000,10,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10397,51,42.400000,18,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10398,35,14.400000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10398,55,19.200000,120,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10399,68,10.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10399,71,17.200000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10399,76,14.400000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10399,77,10.400000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10400,29,99.000000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10400,35,14.400000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10400,49,16.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10401,30,20.700000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10401,56,30.400000,70,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10401,65,16.800000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10401,71,17.200000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10402,23,7.200000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10402,63,35.100000,65,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10403,16,13.900000,21,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10403,48,10.200000,70,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10404,26,24.900000,30,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10404,42,11.200000,40,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10404,49,16.000000,30,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10405,3,8.000000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10406,1,14.400000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10406,21,8.000000,30,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10406,28,36.400000,42,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10406,36,15.200000,5,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10406,40,14.700000,2,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10407,11,16.800000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10407,69,28.800000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10407,71,17.200000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10408,37,20.800000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10408,54,5.900000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10408,62,39.400000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10409,14,18.600000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10409,21,8.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10410,33,2.000000,49,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10410,59,44.000000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10411,41,7.700000,25,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10411,44,15.500000,40,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10411,59,44.000000,9,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10412,14,18.600000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10413,1,14.400000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10413,62,39.400000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10413,76,14.400000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10414,19,7.300000,18,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10414,33,2.000000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10415,17,31.200000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10415,33,2.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10416,19,7.300000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10416,53,26.200000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10416,57,15.600000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10417,38,210.800000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10417,46,9.600000,2,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10417,68,10.000000,36,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10417,77,10.400000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10418,2,15.200000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10418,47,7.600000,55,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10418,61,22.800000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10418,74,8.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10419,60,27.200000,60,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10419,69,28.800000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10420,9,77.600000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10420,13,4.800000,2,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10420,70,12.000000,8,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10420,73,12.000000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10421,19,7.300000,4,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10421,26,24.900000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10421,53,26.200000,15,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10421,77,10.400000,10,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10422,26,24.900000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10423,31,10.000000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10423,59,44.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10424,35,14.400000,60,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10424,38,210.800000,49,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10424,68,10.000000,30,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10425,55,19.200000,10,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10425,76,14.400000,20,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10426,56,30.400000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10426,64,26.600000,7,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10427,14,18.600000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10428,46,9.600000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10429,50,13.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10429,63,35.100000,35,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10430,17,31.200000,45,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10430,21,8.000000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10430,56,30.400000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10430,59,44.000000,70,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10431,17,31.200000,50,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10431,40,14.700000,50,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10431,47,7.600000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10432,26,24.900000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10432,54,5.900000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10433,56,30.400000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10434,11,16.800000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10434,76,14.400000,18,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10435,2,15.200000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10435,22,16.800000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10435,72,27.800000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10436,46,9.600000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10436,56,30.400000,40,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10436,64,26.600000,30,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10436,75,6.200000,24,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10437,53,26.200000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10438,19,7.300000,15,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10438,34,11.200000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10438,57,15.600000,15,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10439,12,30.400000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10439,16,13.900000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10439,64,26.600000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10439,74,8.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10440,2,15.200000,45,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10440,16,13.900000,49,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10440,29,99.000000,24,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10440,61,22.800000,90,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10441,27,35.100000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10442,11,16.800000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10442,54,5.900000,80,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10442,66,13.600000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10443,11,16.800000,6,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10443,28,36.400000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10444,17,31.200000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10444,26,24.900000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10444,35,14.400000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10444,41,7.700000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10445,39,14.400000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10445,54,5.900000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10446,19,7.300000,12,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10446,24,3.600000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10446,31,10.000000,3,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10446,52,5.600000,15,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10447,19,7.300000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10447,65,16.800000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10447,71,17.200000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10448,26,24.900000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10448,40,14.700000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10449,10,24.800000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10449,52,5.600000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10449,62,39.400000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10450,10,24.800000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10450,54,5.900000,6,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10451,55,19.200000,120,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10451,64,26.600000,35,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10451,65,16.800000,28,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10451,77,10.400000,55,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10452,28,36.400000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10452,44,15.500000,100,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10453,48,10.200000,15,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10453,70,12.000000,25,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10454,16,13.900000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10454,33,2.000000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10454,46,9.600000,10,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10455,39,14.400000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10455,53,26.200000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10455,61,22.800000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10455,71,17.200000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10456,21,8.000000,40,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10456,49,16.000000,21,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10457,59,44.000000,36,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10458,26,24.900000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10458,28,36.400000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10458,43,36.800000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10458,56,30.400000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10458,71,17.200000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10459,7,24.000000,16,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10459,46,9.600000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10459,72,27.800000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10460,68,10.000000,21,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10460,75,6.200000,4,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10461,21,8.000000,40,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10461,30,20.700000,28,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10461,55,19.200000,60,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10462,13,4.800000,1,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10462,23,7.200000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10463,19,7.300000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10463,42,11.200000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10464,4,17.600000,16,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10464,43,36.800000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10464,56,30.400000,30,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10464,60,27.200000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10465,24,3.600000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10465,29,99.000000,18,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10465,40,14.700000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10465,45,7.600000,30,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10465,50,13.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10466,11,16.800000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10466,46,9.600000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10467,24,3.600000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10467,25,11.200000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10468,30,20.700000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10468,43,36.800000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10469,2,15.200000,40,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10469,16,13.900000,35,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10469,44,15.500000,2,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10470,18,50.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10470,23,7.200000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10470,64,26.600000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10471,7,24.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10471,56,30.400000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10472,24,3.600000,80,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10472,51,42.400000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10473,33,2.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10473,71,17.200000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10474,14,18.600000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10474,28,36.400000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10474,40,14.700000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10474,75,6.200000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10475,31,10.000000,35,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10475,66,13.600000,60,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10475,76,14.400000,42,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10476,55,19.200000,2,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10476,70,12.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10477,1,14.400000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10477,21,8.000000,21,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10477,39,14.400000,20,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10478,10,24.800000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10479,38,210.800000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10479,53,26.200000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10479,59,44.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10479,64,26.600000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10480,47,7.600000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10480,59,44.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10481,49,16.000000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10481,60,27.200000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10482,40,14.700000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10483,34,11.200000,35,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10483,77,10.400000,30,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10484,21,8.000000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10484,40,14.700000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10484,51,42.400000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10485,2,15.200000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10485,3,8.000000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10485,55,19.200000,30,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10485,70,12.000000,60,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10486,11,16.800000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10486,51,42.400000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10486,74,8.000000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10487,19,7.300000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10487,26,24.900000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10487,54,5.900000,24,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10488,59,44.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10488,73,12.000000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10489,11,16.800000,15,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10489,16,13.900000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10490,59,44.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10490,68,10.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10490,75,6.200000,36,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10491,44,15.500000,15,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10491,77,10.400000,7,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10492,25,11.200000,60,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10492,42,11.200000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10493,65,16.800000,15,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10493,66,13.600000,10,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10493,69,28.800000,10,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10494,56,30.400000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10495,23,7.200000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10495,41,7.700000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10495,77,10.400000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10496,31,10.000000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10497,56,30.400000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10497,72,27.800000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10497,77,10.400000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10498,24,4.500000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10498,40,18.400000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10498,42,14.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10499,28,45.600000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10499,49,20.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10500,15,15.500000,12,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10500,28,45.600000,8,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10501,54,7.450000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10502,45,9.500000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10502,53,32.800000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10502,67,14.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10503,14,23.250000,70,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10503,65,21.050000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10504,2,19.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10504,21,10.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10504,53,32.800000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10504,61,28.500000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10505,62,49.300000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10506,25,14.000000,18,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10506,70,15.000000,14,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10507,43,46.000000,15,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10507,48,12.750000,15,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10508,13,6.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10508,39,18.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10509,28,45.600000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10510,29,123.790000,36,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10510,75,7.750000,36,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10511,4,22.000000,50,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10511,7,30.000000,50,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10511,8,40.000000,10,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10512,24,4.500000,10,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10512,46,12.000000,9,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10512,47,9.500000,6,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10512,60,34.000000,12,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10513,21,10.000000,40,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10513,32,32.000000,50,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10513,61,28.500000,15,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10514,20,81.000000,39,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10514,28,45.600000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10514,56,38.000000,70,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10514,65,21.050000,39,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10514,75,7.750000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10515,9,97.000000,16,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10515,16,17.450000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10515,27,43.900000,120,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10515,33,2.500000,16,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10515,60,34.000000,84,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10516,18,62.500000,25,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10516,41,9.650000,80,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10516,42,14.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10517,52,7.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10517,59,55.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10517,70,15.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10518,24,4.500000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10518,38,263.500000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10518,44,19.450000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10519,10,31.000000,16,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10519,56,38.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10519,60,34.000000,10,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10520,24,4.500000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10520,53,32.800000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10521,35,18.000000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10521,41,9.650000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10521,68,12.500000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10522,1,18.000000,40,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10522,8,40.000000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10522,30,25.890000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10522,40,18.400000,25,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10523,17,39.000000,25,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10523,20,81.000000,15,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10523,37,26.000000,18,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10523,41,9.650000,6,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10524,10,31.000000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10524,30,25.890000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10524,43,46.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10524,54,7.450000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10525,36,19.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10525,40,18.400000,15,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10526,1,18.000000,8,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10526,13,6.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10526,56,38.000000,30,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10527,4,22.000000,50,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10527,36,19.000000,30,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10528,11,21.000000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10528,33,2.500000,8,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10528,72,34.800000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10529,55,24.000000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10529,68,12.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10529,69,36.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10530,17,39.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10530,43,46.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10530,61,28.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10530,76,18.000000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10531,59,55.000000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10532,30,25.890000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10532,66,17.000000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10533,4,22.000000,50,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10533,72,34.800000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10533,73,15.000000,24,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10534,30,25.890000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10534,40,18.400000,10,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10534,54,7.450000,10,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10535,11,21.000000,50,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10535,40,18.400000,10,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10535,57,19.500000,5,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10535,59,55.000000,15,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10536,12,38.000000,15,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10536,31,12.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10536,33,2.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10536,60,34.000000,35,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10537,31,12.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10537,51,53.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10537,58,13.250000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10537,72,34.800000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10537,73,15.000000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10538,70,15.000000,7,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10538,72,34.800000,1,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10539,13,6.000000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10539,21,10.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10539,33,2.500000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10539,49,20.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10540,3,10.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10540,26,31.230000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10540,38,263.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10540,68,12.500000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10541,24,4.500000,35,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10541,38,263.500000,4,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10541,65,21.050000,36,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10541,71,21.500000,9,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10542,11,21.000000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10542,54,7.450000,24,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10543,12,38.000000,30,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10543,23,9.000000,70,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10544,28,45.600000,7,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10544,67,14.000000,7,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10545,11,21.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10546,7,30.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10546,35,18.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10546,62,49.300000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10547,32,32.000000,24,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10547,36,19.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10548,34,14.000000,10,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10548,41,9.650000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10549,31,12.500000,55,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10549,45,9.500000,100,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10549,51,53.000000,48,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10550,17,39.000000,8,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10550,19,9.200000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10550,21,10.000000,6,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10550,61,28.500000,10,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10551,16,17.450000,40,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10551,35,18.000000,20,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10551,44,19.450000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10552,69,36.000000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10552,75,7.750000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10553,11,21.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10553,16,17.450000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10553,22,21.000000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10553,31,12.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10553,35,18.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10554,16,17.450000,30,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10554,23,9.000000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10554,62,49.300000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10554,77,13.000000,10,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10555,14,23.250000,30,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10555,19,9.200000,35,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10555,24,4.500000,18,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10555,51,53.000000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10555,56,38.000000,40,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10556,72,34.800000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10557,64,33.250000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10557,75,7.750000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10558,47,9.500000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10558,51,53.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10558,52,7.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10558,53,32.800000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10558,73,15.000000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10559,41,9.650000,12,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10559,55,24.000000,18,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10560,30,25.890000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10560,62,49.300000,15,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10561,44,19.450000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10561,51,53.000000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10562,33,2.500000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10562,62,49.300000,10,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10563,36,19.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10563,52,7.000000,70,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10564,17,39.000000,16,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10564,31,12.500000,6,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10564,55,24.000000,25,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10565,24,4.500000,25,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10565,64,33.250000,18,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10566,11,21.000000,35,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10566,18,62.500000,18,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10566,76,18.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10567,31,12.500000,60,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10567,51,53.000000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10567,59,55.000000,40,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10568,10,31.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10569,31,12.500000,35,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10569,76,18.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10570,11,21.000000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10570,56,38.000000,60,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10571,14,23.250000,11,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10571,42,14.000000,28,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10572,16,17.450000,12,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10572,32,32.000000,10,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10572,40,18.400000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10572,75,7.750000,15,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10573,17,39.000000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10573,34,14.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10573,53,32.800000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10574,33,2.500000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10574,40,18.400000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10574,62,49.300000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10574,64,33.250000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10575,59,55.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10575,63,43.900000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10575,72,34.800000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10575,76,18.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10576,1,18.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10576,31,12.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10576,44,19.450000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10577,39,18.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10577,75,7.750000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10577,77,13.000000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10578,35,18.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10578,57,19.500000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10579,15,15.500000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10579,75,7.750000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10580,14,23.250000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10580,41,9.650000,9,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10580,65,21.050000,30,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10581,75,7.750000,50,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10582,57,19.500000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10582,76,18.000000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10583,29,123.790000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10583,60,34.000000,24,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10583,69,36.000000,10,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10584,31,12.500000,50,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10585,47,9.500000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10586,52,7.000000,4,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10587,26,31.230000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10587,35,18.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10587,77,13.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10588,18,62.500000,40,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10588,42,14.000000,100,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10589,35,18.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10590,1,18.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10590,77,13.000000,60,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10591,3,10.000000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10591,7,30.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10591,54,7.450000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10592,15,15.500000,25,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10592,26,31.230000,5,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10593,20,81.000000,21,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10593,69,36.000000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10593,76,18.000000,4,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10594,52,7.000000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10594,58,13.250000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10595,35,18.000000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10595,61,28.500000,120,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10595,69,36.000000,65,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10596,56,38.000000,5,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10596,63,43.900000,24,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10596,75,7.750000,30,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10597,24,4.500000,35,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10597,57,19.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10597,65,21.050000,12,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10598,27,43.900000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10598,71,21.500000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10599,62,49.300000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10600,54,7.450000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10600,73,15.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10601,13,6.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10601,59,55.000000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10602,77,13.000000,5,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10603,22,21.000000,48,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10603,49,20.000000,25,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10604,48,12.750000,6,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10604,76,18.000000,10,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10605,16,17.450000,30,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10605,59,55.000000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10605,60,34.000000,70,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10605,71,21.500000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10606,4,22.000000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10606,55,24.000000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10606,62,49.300000,10,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10607,7,30.000000,45,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10607,17,39.000000,100,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10607,33,2.500000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10607,40,18.400000,42,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10607,72,34.800000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10608,56,38.000000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10609,1,18.000000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10609,10,31.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10609,21,10.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10610,36,19.000000,21,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10611,1,18.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10611,2,19.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10611,60,34.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10612,10,31.000000,70,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10612,36,19.000000,55,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10612,49,20.000000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10612,60,34.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10612,76,18.000000,80,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10613,13,6.000000,8,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10613,75,7.750000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10614,11,21.000000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10614,21,10.000000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10614,39,18.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10615,55,24.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10616,38,263.500000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10616,56,38.000000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10616,70,15.000000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10616,71,21.500000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10617,59,55.000000,30,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10618,6,25.000000,70,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10618,56,38.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10618,68,12.500000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10619,21,10.000000,42,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10619,22,21.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10620,24,4.500000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10620,52,7.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10621,19,9.200000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10621,23,9.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10621,70,15.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10621,71,21.500000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10622,2,19.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10622,68,12.500000,18,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10623,14,23.250000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10623,19,9.200000,15,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10623,21,10.000000,25,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10623,24,4.500000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10623,35,18.000000,30,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10624,28,45.600000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10624,29,123.790000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10624,44,19.450000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10625,14,23.250000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10625,42,14.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10625,60,34.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10626,53,32.800000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10626,60,34.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10626,71,21.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10627,62,49.300000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10627,73,15.000000,35,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10628,1,18.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10629,29,123.790000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10629,64,33.250000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10630,55,24.000000,12,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10630,76,18.000000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10631,75,7.750000,8,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10632,2,19.000000,30,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10632,33,2.500000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10633,12,38.000000,36,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10633,13,6.000000,13,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10633,26,31.230000,35,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10633,62,49.300000,80,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10634,7,30.000000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10634,18,62.500000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10634,51,53.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10634,75,7.750000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10635,4,22.000000,10,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10635,5,21.350000,15,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10635,22,21.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10636,4,22.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10636,58,13.250000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10637,11,21.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10637,50,16.250000,25,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10637,56,38.000000,60,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10638,45,9.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10638,65,21.050000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10638,72,34.800000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10639,18,62.500000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10640,69,36.000000,20,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10640,70,15.000000,15,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10641,2,19.000000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10641,40,18.400000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10642,21,10.000000,30,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10642,61,28.500000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10643,28,45.600000,15,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10643,39,18.000000,21,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10643,46,12.000000,2,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10644,18,62.500000,4,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10644,43,46.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10644,46,12.000000,21,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10645,18,62.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10645,36,19.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10646,1,18.000000,15,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10646,10,31.000000,18,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10646,71,21.500000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10646,77,13.000000,35,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10647,19,9.200000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10647,39,18.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10648,22,21.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10648,24,4.500000,15,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10649,28,45.600000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10649,72,34.800000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10650,30,25.890000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10650,53,32.800000,25,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10650,54,7.450000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10651,19,9.200000,12,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10651,22,21.000000,20,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10652,30,25.890000,2,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10652,42,14.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10653,16,17.450000,30,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10653,60,34.000000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10654,4,22.000000,12,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10654,39,18.000000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10654,54,7.450000,6,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10655,41,9.650000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10656,14,23.250000,3,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10656,44,19.450000,28,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10656,47,9.500000,6,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10657,15,15.500000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10657,41,9.650000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10657,46,12.000000,45,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10657,47,9.500000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10657,56,38.000000,45,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10657,60,34.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10658,21,10.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10658,40,18.400000,70,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10658,60,34.000000,55,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10658,77,13.000000,70,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10659,31,12.500000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10659,40,18.400000,24,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10659,70,15.000000,40,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10660,20,81.000000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10661,39,18.000000,3,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10661,58,13.250000,49,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10662,68,12.500000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10663,40,18.400000,30,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10663,42,14.000000,30,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10663,51,53.000000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10664,10,31.000000,24,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10664,56,38.000000,12,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10664,65,21.050000,15,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10665,51,53.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10665,59,55.000000,1,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10665,76,18.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10666,29,123.790000,36,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10666,65,21.050000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10667,69,36.000000,45,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10667,71,21.500000,14,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10668,31,12.500000,8,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10668,55,24.000000,4,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10668,64,33.250000,15,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10669,36,19.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10670,23,9.000000,32,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10670,46,12.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10670,67,14.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10670,73,15.000000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10670,75,7.750000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10671,16,17.450000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10671,62,49.300000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10671,65,21.050000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10672,38,263.500000,15,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10672,71,21.500000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10673,16,17.450000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10673,42,14.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10673,43,46.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10674,23,9.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10675,14,23.250000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10675,53,32.800000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10675,58,13.250000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10676,10,31.000000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10676,19,9.200000,7,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10676,44,19.450000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10677,26,31.230000,30,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10677,33,2.500000,8,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10678,12,38.000000,100,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10678,33,2.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10678,41,9.650000,120,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10678,54,7.450000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10679,59,55.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10680,16,17.450000,50,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10680,31,12.500000,20,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10680,42,14.000000,40,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10681,19,9.200000,30,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10681,21,10.000000,12,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10681,64,33.250000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10682,33,2.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10682,66,17.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10682,75,7.750000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10683,52,7.000000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10684,40,18.400000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10684,47,9.500000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10684,60,34.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10685,10,31.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10685,41,9.650000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10685,47,9.500000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10686,17,39.000000,30,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10686,26,31.230000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10687,9,97.000000,50,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10687,29,123.790000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10687,36,19.000000,6,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10688,10,31.000000,18,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10688,28,45.600000,60,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10688,34,14.000000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10689,1,18.000000,35,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10690,56,38.000000,20,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10690,77,13.000000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10691,1,18.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10691,29,123.790000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10691,43,46.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10691,44,19.450000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10691,62,49.300000,48,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10692,63,43.900000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10693,9,97.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10693,54,7.450000,60,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10693,69,36.000000,30,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10693,73,15.000000,15,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10694,7,30.000000,90,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10694,59,55.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10694,70,15.000000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10695,8,40.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10695,12,38.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10695,24,4.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10696,17,39.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10696,46,12.000000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10697,19,9.200000,7,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10697,35,18.000000,9,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10697,58,13.250000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10697,70,15.000000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10698,11,21.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10698,17,39.000000,8,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10698,29,123.790000,12,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10698,65,21.050000,65,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10698,70,15.000000,8,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10699,47,9.500000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10700,1,18.000000,5,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10700,34,14.000000,12,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10700,68,12.500000,40,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10700,71,21.500000,60,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10701,59,55.000000,42,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10701,71,21.500000,20,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10701,76,18.000000,35,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10702,3,10.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10702,76,18.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10703,2,19.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10703,59,55.000000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10703,73,15.000000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10704,4,22.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10704,24,4.500000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10704,48,12.750000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10705,31,12.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10705,32,32.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10706,16,17.450000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10706,43,46.000000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10706,59,55.000000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10707,55,24.000000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10707,57,19.500000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10707,70,15.000000,28,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10708,5,21.350000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10708,36,19.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10709,8,40.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10709,51,53.000000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10709,60,34.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10710,19,9.200000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10710,47,9.500000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10711,19,9.200000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10711,41,9.650000,42,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10711,53,32.800000,120,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10712,53,32.800000,3,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10712,56,38.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10713,10,31.000000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10713,26,31.230000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10713,45,9.500000,110,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10713,46,12.000000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10714,2,19.000000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10714,17,39.000000,27,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10714,47,9.500000,50,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10714,56,38.000000,18,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10714,58,13.250000,12,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10715,10,31.000000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10715,71,21.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10716,21,10.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10716,51,53.000000,7,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10716,61,28.500000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10717,21,10.000000,32,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10717,54,7.450000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10717,69,36.000000,25,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10718,12,38.000000,36,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10718,16,17.450000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10718,36,19.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10718,62,49.300000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10719,18,62.500000,12,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10719,30,25.890000,3,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10719,54,7.450000,40,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10720,35,18.000000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10720,71,21.500000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10721,44,19.450000,50,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10722,2,19.000000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10722,31,12.500000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10722,68,12.500000,45,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10722,75,7.750000,42,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10723,26,31.230000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10724,10,31.000000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10724,61,28.500000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10725,41,9.650000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10725,52,7.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10725,55,24.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10726,4,22.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10726,11,21.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10727,17,39.000000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10727,56,38.000000,10,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10727,59,55.000000,10,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10728,30,25.890000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10728,40,18.400000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10728,55,24.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10728,60,34.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10729,1,18.000000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10729,21,10.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10729,50,16.250000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10730,16,17.450000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10730,31,12.500000,3,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10730,65,21.050000,10,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10731,21,10.000000,40,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10731,51,53.000000,30,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10732,76,18.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10733,14,23.250000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10733,28,45.600000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10733,52,7.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10734,6,25.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10734,30,25.890000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10734,76,18.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10735,61,28.500000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10735,77,13.000000,2,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10736,65,21.050000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10736,75,7.750000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10737,13,6.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10737,41,9.650000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10738,16,17.450000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10739,36,19.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10739,52,7.000000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10740,28,45.600000,5,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10740,35,18.000000,35,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10740,45,9.500000,40,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10740,56,38.000000,14,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10741,2,19.000000,15,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10742,3,10.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10742,60,34.000000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10742,72,34.800000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10743,46,12.000000,28,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10744,40,18.400000,50,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10745,18,62.500000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10745,44,19.450000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10745,59,55.000000,45,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10745,72,34.800000,7,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10746,13,6.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10746,42,14.000000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10746,62,49.300000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10746,69,36.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10747,31,12.500000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10747,41,9.650000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10747,63,43.900000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10747,69,36.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10748,23,9.000000,44,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10748,40,18.400000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10748,56,38.000000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10749,56,38.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10749,59,55.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10749,76,18.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10750,14,23.250000,5,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10750,45,9.500000,40,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10750,59,55.000000,25,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10751,26,31.230000,12,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10751,30,25.890000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10751,50,16.250000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10751,73,15.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10752,1,18.000000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10752,69,36.000000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10753,45,9.500000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10753,74,10.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10754,40,18.400000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10755,47,9.500000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10755,56,38.000000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10755,57,19.500000,14,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10755,69,36.000000,25,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10756,18,62.500000,21,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10756,36,19.000000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10756,68,12.500000,6,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10756,69,36.000000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10757,34,14.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10757,59,55.000000,7,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10757,62,49.300000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10757,64,33.250000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10758,26,31.230000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10758,52,7.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10758,70,15.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10759,32,32.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10760,25,14.000000,12,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10760,27,43.900000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10760,43,46.000000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10761,25,14.000000,35,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10761,75,7.750000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10762,39,18.000000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10762,47,9.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10762,51,53.000000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10762,56,38.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10763,21,10.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10763,22,21.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10763,24,4.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10764,3,10.000000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10764,39,18.000000,130,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10765,65,21.050000,80,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10766,2,19.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10766,7,30.000000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10766,68,12.500000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10767,42,14.000000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10768,22,21.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10768,31,12.500000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10768,60,34.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10768,71,21.500000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10769,41,9.650000,30,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10769,52,7.000000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10769,61,28.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10769,62,49.300000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10770,11,21.000000,15,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10771,71,21.500000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10772,29,123.790000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10772,59,55.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10773,17,39.000000,33,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10773,31,12.500000,70,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10773,75,7.750000,7,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10774,31,12.500000,2,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10774,66,17.000000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10775,10,31.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10775,67,14.000000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10776,31,12.500000,16,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10776,42,14.000000,12,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10776,45,9.500000,27,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10776,51,53.000000,120,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10777,42,14.000000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10778,41,9.650000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10779,16,17.450000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10779,62,49.300000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10780,70,15.000000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10780,77,13.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10781,54,7.450000,3,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10781,56,38.000000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10781,74,10.000000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10782,31,12.500000,1,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10783,31,12.500000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10783,38,263.500000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10784,36,19.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10784,39,18.000000,2,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10784,72,34.800000,30,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10785,10,31.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10785,75,7.750000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10786,8,40.000000,30,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10786,30,25.890000,15,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10786,75,7.750000,42,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10787,2,19.000000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10787,29,123.790000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10788,19,9.200000,50,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10788,75,7.750000,40,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10789,18,62.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10789,35,18.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10789,63,43.900000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10789,68,12.500000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10790,7,30.000000,3,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10790,56,38.000000,20,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10791,29,123.790000,14,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10791,41,9.650000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10792,2,19.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10792,54,7.450000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10792,68,12.500000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10793,41,9.650000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10793,52,7.000000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10794,14,23.250000,15,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10794,54,7.450000,6,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10795,16,17.450000,65,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10795,17,39.000000,35,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10796,26,31.230000,21,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10796,44,19.450000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10796,64,33.250000,35,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10796,69,36.000000,24,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10797,11,21.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10798,62,49.300000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10798,72,34.800000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10799,13,6.000000,20,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10799,24,4.500000,20,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10799,59,55.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10800,11,21.000000,50,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10800,51,53.000000,10,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10800,54,7.450000,7,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10801,17,39.000000,40,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10801,29,123.790000,20,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10802,30,25.890000,25,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10802,51,53.000000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10802,55,24.000000,60,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10802,62,49.300000,5,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10803,19,9.200000,24,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10803,25,14.000000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10803,59,55.000000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10804,10,31.000000,36,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10804,28,45.600000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10804,49,20.000000,4,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10805,34,14.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10805,38,263.500000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10806,2,19.000000,20,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10806,65,21.050000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10806,74,10.000000,15,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10807,40,18.400000,1,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10808,56,38.000000,20,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10808,76,18.000000,50,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10809,52,7.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10810,13,6.000000,7,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10810,25,14.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10810,70,15.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10811,19,9.200000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10811,23,9.000000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10811,40,18.400000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10812,31,12.500000,16,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10812,72,34.800000,40,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10812,77,13.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10813,2,19.000000,12,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10813,46,12.000000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10814,41,9.650000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10814,43,46.000000,20,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10814,48,12.750000,8,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10814,61,28.500000,30,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10815,33,2.500000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10816,38,263.500000,30,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10816,62,49.300000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10817,26,31.230000,40,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10817,38,263.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10817,40,18.400000,60,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10817,62,49.300000,25,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10818,32,32.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10818,41,9.650000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10819,43,46.000000,7,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10819,75,7.750000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10820,56,38.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10821,35,18.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10821,51,53.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10822,62,49.300000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10822,70,15.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10823,11,21.000000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10823,57,19.500000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10823,59,55.000000,40,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10823,77,13.000000,15,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10824,41,9.650000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10824,70,15.000000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10825,26,31.230000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10825,53,32.800000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10826,31,12.500000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10826,57,19.500000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10827,10,31.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10827,39,18.000000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10828,20,81.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10828,38,263.500000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10829,2,19.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10829,8,40.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10829,13,6.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10829,60,34.000000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10830,6,25.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10830,39,18.000000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10830,60,34.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10830,68,12.500000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10831,19,9.200000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10831,35,18.000000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10831,38,263.500000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10831,43,46.000000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10832,13,6.000000,3,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10832,25,14.000000,10,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10832,44,19.450000,16,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10832,64,33.250000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10833,7,30.000000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10833,31,12.500000,9,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10833,53,32.800000,9,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10834,29,123.790000,8,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10834,30,25.890000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10835,59,55.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10835,77,13.000000,2,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10836,22,21.000000,52,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10836,35,18.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10836,57,19.500000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10836,60,34.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10836,64,33.250000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10837,13,6.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10837,40,18.400000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10837,47,9.500000,40,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10837,76,18.000000,21,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10838,1,18.000000,4,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10838,18,62.500000,25,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10838,36,19.000000,50,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10839,58,13.250000,30,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10839,72,34.800000,15,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10840,25,14.000000,6,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10840,39,18.000000,10,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10841,10,31.000000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10841,56,38.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10841,59,55.000000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10841,77,13.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10842,11,21.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10842,43,46.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10842,68,12.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10842,70,15.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10843,51,53.000000,4,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10844,22,21.000000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10845,23,9.000000,70,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10845,35,18.000000,25,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10845,42,14.000000,42,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10845,58,13.250000,60,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10845,64,33.250000,48,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10846,4,22.000000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10846,70,15.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10846,74,10.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10847,1,18.000000,80,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10847,19,9.200000,12,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10847,37,26.000000,60,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10847,45,9.500000,36,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10847,60,34.000000,45,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10847,71,21.500000,55,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10848,5,21.350000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10848,9,97.000000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10849,3,10.000000,49,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10849,26,31.230000,18,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10850,25,14.000000,20,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10850,33,2.500000,4,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10850,70,15.000000,30,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10851,2,19.000000,5,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10851,25,14.000000,10,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10851,57,19.500000,10,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10851,59,55.000000,42,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10852,2,19.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10852,17,39.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10852,62,49.300000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10853,18,62.500000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10854,10,31.000000,100,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10854,13,6.000000,65,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10855,16,17.450000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10855,31,12.500000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10855,56,38.000000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10855,65,21.050000,15,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10856,2,19.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10856,42,14.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10857,3,10.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10857,26,31.230000,35,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10857,29,123.790000,10,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10858,7,30.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10858,27,43.900000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10858,70,15.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10859,24,4.500000,40,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10859,54,7.450000,35,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10859,64,33.250000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10860,51,53.000000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10860,76,18.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10861,17,39.000000,42,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10861,18,62.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10861,21,10.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10861,33,2.500000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10861,62,49.300000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10862,11,21.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10862,52,7.000000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10863,1,18.000000,20,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10863,58,13.250000,12,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10864,35,18.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10864,67,14.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10865,38,263.500000,60,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10865,39,18.000000,80,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10866,2,19.000000,21,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10866,24,4.500000,6,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10866,30,25.890000,40,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10867,53,32.800000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10868,26,31.230000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10868,35,18.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10868,49,20.000000,42,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10869,1,18.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10869,11,21.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10869,23,9.000000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10869,68,12.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10870,35,18.000000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10870,51,53.000000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10871,6,25.000000,50,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10871,16,17.450000,12,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10871,17,39.000000,16,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10872,55,24.000000,10,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10872,62,49.300000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10872,64,33.250000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10872,65,21.050000,21,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10873,21,10.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10873,28,45.600000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10874,10,31.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10875,19,9.200000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10875,47,9.500000,21,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10875,49,20.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10876,46,12.000000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10876,64,33.250000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10877,16,17.450000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10877,18,62.500000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10878,20,81.000000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10879,40,18.400000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10879,65,21.050000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10879,76,18.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10880,23,9.000000,30,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10880,61,28.500000,30,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10880,70,15.000000,50,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10881,73,15.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10882,42,14.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10882,49,20.000000,20,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10882,54,7.450000,32,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10883,24,4.500000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10884,21,10.000000,40,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10884,56,38.000000,21,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10884,65,21.050000,12,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10885,2,19.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10885,24,4.500000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10885,70,15.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10885,77,13.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10886,10,31.000000,70,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10886,31,12.500000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10886,77,13.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10887,25,14.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10888,2,19.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10888,68,12.500000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10889,11,21.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10889,38,263.500000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10890,17,39.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10890,34,14.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10890,41,9.650000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10891,30,25.890000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10892,59,55.000000,40,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10893,8,40.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10893,24,4.500000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10893,29,123.790000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10893,30,25.890000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10893,36,19.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10894,13,6.000000,28,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10894,69,36.000000,50,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10894,75,7.750000,120,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10895,24,4.500000,110,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10895,39,18.000000,45,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10895,40,18.400000,91,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10895,60,34.000000,100,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10896,45,9.500000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10896,56,38.000000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10897,29,123.790000,80,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10897,30,25.890000,36,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10898,13,6.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10899,39,18.000000,8,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10900,70,15.000000,3,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10901,41,9.650000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10901,71,21.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10902,55,24.000000,30,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10902,62,49.300000,6,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10903,13,6.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10903,65,21.050000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10903,68,12.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10904,58,13.250000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10904,62,49.300000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10905,1,18.000000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10906,61,28.500000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10907,75,7.750000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10908,7,30.000000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10908,52,7.000000,14,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10909,7,30.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10909,16,17.450000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10909,41,9.650000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10910,19,9.200000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10910,49,20.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10910,61,28.500000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10911,1,18.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10911,17,39.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10911,67,14.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10912,11,21.000000,40,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10912,29,123.790000,60,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10913,4,22.000000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10913,33,2.500000,40,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10913,58,13.250000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10914,71,21.500000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10915,17,39.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10915,33,2.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10915,54,7.450000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10916,16,17.450000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10916,32,32.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10916,57,19.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10917,30,25.890000,1,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10917,60,34.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10918,1,18.000000,60,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10918,60,34.000000,25,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10919,16,17.450000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10919,25,14.000000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10919,40,18.400000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10920,50,16.250000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10921,35,18.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10921,63,43.900000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10922,17,39.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10922,24,4.500000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10923,42,14.000000,10,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10923,43,46.000000,10,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10923,67,14.000000,24,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10924,10,31.000000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10924,28,45.600000,30,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10924,75,7.750000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10925,36,19.000000,25,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10925,52,7.000000,12,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10926,11,21.000000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10926,13,6.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10926,19,9.200000,7,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10926,72,34.800000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10927,20,81.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10927,52,7.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10927,76,18.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10928,47,9.500000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10928,76,18.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10929,21,10.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10929,75,7.750000,49,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10929,77,13.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10930,21,10.000000,36,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10930,27,43.900000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10930,55,24.000000,25,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10930,58,13.250000,30,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10931,13,6.000000,42,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10931,57,19.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10932,16,17.450000,30,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10932,62,49.300000,14,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10932,72,34.800000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10932,75,7.750000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10933,53,32.800000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10933,61,28.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10934,6,25.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10935,1,18.000000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10935,18,62.500000,4,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10935,23,9.000000,8,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10936,36,19.000000,30,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10937,28,45.600000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10937,34,14.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10938,13,6.000000,20,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10938,43,46.000000,24,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10938,60,34.000000,49,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10938,71,21.500000,35,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10939,2,19.000000,10,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10939,67,14.000000,40,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10940,7,30.000000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10940,13,6.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10941,31,12.500000,44,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10941,62,49.300000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10941,68,12.500000,80,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10941,72,34.800000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10942,49,20.000000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10943,13,6.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10943,22,21.000000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10943,46,12.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10944,11,21.000000,5,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10944,44,19.450000,18,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10944,56,38.000000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10945,13,6.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10945,31,12.500000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10946,10,31.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10946,24,4.500000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10946,77,13.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10947,59,55.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10948,50,16.250000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10948,51,53.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10948,55,24.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10949,6,25.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10949,10,31.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10949,17,39.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10949,62,49.300000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10950,4,22.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10951,33,2.500000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10951,41,9.650000,6,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10951,75,7.750000,50,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10952,6,25.000000,16,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10952,28,45.600000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10953,20,81.000000,50,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10953,31,12.500000,50,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10954,16,17.450000,28,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10954,31,12.500000,25,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10954,45,9.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10954,60,34.000000,24,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10955,75,7.750000,12,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10956,21,10.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10956,47,9.500000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10956,51,53.000000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10957,30,25.890000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10957,35,18.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10957,64,33.250000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10958,5,21.350000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10958,7,30.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10958,72,34.800000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10959,75,7.750000,20,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10960,24,4.500000,10,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10960,41,9.650000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10961,52,7.000000,6,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10961,76,18.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10962,7,30.000000,45,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10962,13,6.000000,77,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10962,53,32.800000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10962,69,36.000000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10962,76,18.000000,44,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10963,60,34.000000,2,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10964,18,62.500000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10964,38,263.500000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10964,69,36.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10965,51,53.000000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10966,37,26.000000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10966,56,38.000000,12,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10966,62,49.300000,12,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10967,19,9.200000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10967,49,20.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10968,12,38.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10968,24,4.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10968,64,33.250000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10969,46,12.000000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10970,52,7.000000,40,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10971,29,123.790000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10972,17,39.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10972,33,2.500000,7,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10973,26,31.230000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10973,41,9.650000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10973,75,7.750000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10974,63,43.900000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10975,8,40.000000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10975,75,7.750000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10976,28,45.600000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10977,39,18.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10977,47,9.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10977,51,53.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10977,63,43.900000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10978,8,40.000000,20,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10978,21,10.000000,40,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10978,40,18.400000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10978,44,19.450000,6,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10979,7,30.000000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10979,12,38.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10979,24,4.500000,80,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10979,27,43.900000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10979,31,12.500000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10979,63,43.900000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10980,75,7.750000,40,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10981,38,263.500000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10982,7,30.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10982,43,46.000000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10983,13,6.000000,84,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10983,57,19.500000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10984,16,17.450000,55,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10984,24,4.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10984,36,19.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10985,16,17.450000,36,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10985,18,62.500000,8,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10985,32,32.000000,35,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10986,11,21.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10986,20,81.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10986,76,18.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10986,77,13.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10987,7,30.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10987,43,46.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10987,72,34.800000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10988,7,30.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10988,62,49.300000,40,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10989,6,25.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10989,11,21.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10989,41,9.650000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10990,21,10.000000,65,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10990,34,14.000000,60,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10990,55,24.000000,65,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10990,61,28.500000,66,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10991,2,19.000000,50,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10991,70,15.000000,20,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10991,76,18.000000,90,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10992,72,34.800000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10993,29,123.790000,50,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10993,41,9.650000,35,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10994,59,55.000000,18,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10995,51,53.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10995,60,34.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10996,42,14.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10997,32,32.000000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10997,46,12.000000,20,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10997,52,7.000000,20,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10998,24,4.500000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10998,61,28.500000,7,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10998,74,10.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10998,75,7.750000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10999,41,9.650000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10999,51,53.000000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(10999,77,13.000000,21,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11000,4,22.000000,25,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11000,24,4.500000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11000,77,13.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11001,7,30.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11001,22,21.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11001,46,12.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11001,55,24.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11002,13,6.000000,56,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11002,35,18.000000,15,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11002,42,14.000000,24,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11002,55,24.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11003,1,18.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11003,40,18.400000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11003,52,7.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11004,26,31.230000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11004,76,18.000000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11005,1,18.000000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11005,59,55.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11006,1,18.000000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11006,29,123.790000,2,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11007,8,40.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11007,29,123.790000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11007,42,14.000000,14,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11008,28,45.600000,70,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11008,34,14.000000,90,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11008,71,21.500000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11009,24,4.500000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11009,36,19.000000,18,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11009,60,34.000000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11010,7,30.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11010,24,4.500000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11011,58,13.250000,40,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11011,71,21.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11012,19,9.200000,50,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11012,60,34.000000,36,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11012,71,21.500000,60,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11013,23,9.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11013,42,14.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11013,45,9.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11013,68,12.500000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11014,41,9.650000,28,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11015,30,25.890000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11015,77,13.000000,18,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11016,31,12.500000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11016,36,19.000000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11017,3,10.000000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11017,59,55.000000,110,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11017,70,15.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11018,12,38.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11018,18,62.500000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11018,56,38.000000,5,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11019,46,12.000000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11019,49,20.000000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11020,10,31.000000,24,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11021,2,19.000000,11,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11021,20,81.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11021,26,31.230000,63,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11021,51,53.000000,44,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11021,72,34.800000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11022,19,9.200000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11022,69,36.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11023,7,30.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11023,43,46.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11024,26,31.230000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11024,33,2.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11024,65,21.050000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11024,71,21.500000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11025,1,18.000000,10,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11025,13,6.000000,20,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11026,18,62.500000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11026,51,53.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11027,24,4.500000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11027,62,49.300000,21,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11028,55,24.000000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11028,59,55.000000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11029,56,38.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11029,63,43.900000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11030,2,19.000000,100,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11030,5,21.350000,70,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11030,29,123.790000,60,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11030,59,55.000000,100,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11031,1,18.000000,45,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11031,13,6.000000,80,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11031,24,4.500000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11031,64,33.250000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11031,71,21.500000,16,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11032,36,19.000000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11032,38,263.500000,25,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11032,59,55.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11033,53,32.800000,70,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11033,69,36.000000,36,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11034,21,10.000000,15,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11034,44,19.450000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11034,61,28.500000,6,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11035,1,18.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11035,35,18.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11035,42,14.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11035,54,7.450000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11036,13,6.000000,7,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11036,59,55.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11037,70,15.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11038,40,18.400000,5,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11038,52,7.000000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11038,71,21.500000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11039,28,45.600000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11039,35,18.000000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11039,49,20.000000,60,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11039,57,19.500000,28,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11040,21,10.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11041,2,19.000000,30,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11041,63,43.900000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11042,44,19.450000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11042,61,28.500000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11043,11,21.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11044,62,49.300000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11045,33,2.500000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11045,51,53.000000,24,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11046,12,38.000000,20,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11046,32,32.000000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11046,35,18.000000,18,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11047,1,18.000000,25,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11047,5,21.350000,30,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11048,68,12.500000,42,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11049,2,19.000000,10,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11049,12,38.000000,4,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11050,76,18.000000,50,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11051,24,4.500000,10,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11052,43,46.000000,30,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11052,61,28.500000,10,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11053,18,62.500000,35,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11053,32,32.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11053,64,33.250000,25,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11054,33,2.500000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11054,67,14.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11055,24,4.500000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11055,25,14.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11055,51,53.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11055,57,19.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11056,7,30.000000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11056,55,24.000000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11056,60,34.000000,50,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11057,70,15.000000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11058,21,10.000000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11058,60,34.000000,21,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11058,61,28.500000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11059,13,6.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11059,17,39.000000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11059,60,34.000000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11060,60,34.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11060,77,13.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11061,60,34.000000,15,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11062,53,32.800000,10,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11062,70,15.000000,12,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11063,34,14.000000,30,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11063,40,18.400000,40,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11063,41,9.650000,30,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11064,17,39.000000,77,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11064,41,9.650000,12,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11064,53,32.800000,25,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11064,55,24.000000,4,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11064,68,12.500000,55,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11065,30,25.890000,4,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11065,54,7.450000,20,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11066,16,17.450000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11066,19,9.200000,42,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11066,34,14.000000,35,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11067,41,9.650000,9,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11068,28,45.600000,8,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11068,43,46.000000,36,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11068,77,13.000000,28,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11069,39,18.000000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11070,1,18.000000,40,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11070,2,19.000000,20,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11070,16,17.450000,30,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11070,31,12.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11071,7,30.000000,15,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11071,13,6.000000,10,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11072,2,19.000000,8,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11072,41,9.650000,40,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11072,50,16.250000,22,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11072,64,33.250000,130,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11073,11,21.000000,10,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11073,24,4.500000,20,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11074,16,17.450000,14,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11075,2,19.000000,10,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11075,46,12.000000,30,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11075,76,18.000000,2,0.150000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11076,6,25.000000,20,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11076,14,23.250000,20,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11076,19,9.200000,10,0.250000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,2,19.000000,24,0.200000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,3,10.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,4,22.000000,1,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,6,25.000000,1,0.020000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,7,30.000000,1,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,8,40.000000,2,0.100000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,10,31.000000,1,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,12,38.000000,2,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,13,6.000000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,14,23.250000,1,0.030000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,16,17.450000,2,0.030000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,20,81.000000,1,0.040000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,23,9.000000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,32,32.000000,1,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,39,18.000000,2,0.050000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,41,9.650000,3,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,46,12.000000,3,0.020000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,52,7.000000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,55,24.000000,2,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,60,34.000000,2,0.060000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,64,33.250000,2,0.030000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,66,17.000000,1,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,73,15.000000,2,0.010000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,75,7.750000,4,0.000000);
INSERT soft Order_Details(OrderID,ProductID,UnitPrice,Quantity,Discount) VALUES(11077,77,13.000000,2,0.000000);

select count(*) from Order_Details;
--ECHO BOTH $IF $EQU $LAST[1] 2155 "PASSED" "***FAILED";
--SET ARGV[$LIF] $+ $ARGV[$LIF] 1;
--ECHO BOTH ": " $LAST[1] " Order Details loaded\n";

--ECHO BOTH "Loading Demo Database Complete\n";

create procedure DB.DBA.rd_v_1_localize_demo()
{
  declare file_text, uriqa varchar;
  uriqa := registry_get('URIQADefaultHost');
  file_text := (select blob_to_string (RES_CONTENT) from WS.WS.SYS_DAV_RES 
    where RES_FULL_PATH='/DAV/VAD/demo/sql/rd_v_1.sql');
  file_text := replace(file_text, 'URIQA2_MACRO', uriqa);
  update WS.WS.SYS_DAV_RES set RES_CONTENT=file_text where RES_FULL_PATH='/DAV/VAD/demo/sql/rd_v_1.sql';
}
;

DB.DBA.rd_v_1_localize_demo()
;

drop procedure DB.DBA.rd_v_1_localize_demo
;
